/*
 * filetransfer.c: The video file transfer facilities
 *
 * See the main source file 'vdr.c' for copyright information and
 * how to reach the author.
 *
 * $Id: $
 */

#include "videodir.h"
#include "filetransfer.h"

static cString StripLastDirectory(const char *DirName)
{
  if (DirName && *DirName) {
     cString s(DirName);
     int l = strlen(*s);
     const char *p = *s + l;
     while (l > 0) {
           if (*p-- == '/')
              break;
           l--;
           }
     if (l)
        s = s.Truncate(l);
     return s;
     }
  return NULL;
}

// --- cCopyingThread --------------------------------------------------------

class cCopyingThread : public cThread {
private:
  const char *error;
  bool deleteSource;
  cString source;
  cString target;
protected:
  virtual void Action(void);
public:
  cCopyingThread(const char *SourceName, const char *ToFileName, bool DeleteSource = false);
  virtual ~cCopyingThread();
  const char *Error(void) { return error; }
  };

cCopyingThread::cCopyingThread(const char *SourceName, const char *TargetName, bool DeleteSource)
:cThread("copying"),
 error(NULL),
 deleteSource(DeleteSource),
 source(SourceName),
 target(TargetName)
{
  // add missing directory delimiters
  const char *delim = "/";
  if (!endswith(*source, delim))
     source = cString::sprintf("%s%s", *source, delim);
  if (!endswith(*target, delim))
     target = cString::sprintf("%s%s", *target, delim);

  Start();
}

cCopyingThread::~cCopyingThread()
{
  Cancel(3);
}

void cCopyingThread::Action(void)
{
  SetPriority(19);
  SetIOPriority(7);

  if (strcmp(*source, *target)) {
     // validate target directory
     if (strstr(*target, *source)) {
        error = "invalid target";
        return;
        }

     // recordings methods require the last directory delimiter to be stripped off
     cString recname = target;
     recname.Truncate(strlen(*recname) - 1);
     Recordings.AddByName(*recname, false);

     RemoveFileOrDir(*target);
     if (!MakeDirs(*target, true)) {
        error = "MakeDirs";
        return;
        }

     if (deleteSource && EntriesOnSameFileSystem(*source, *target)) {
        if (rename(*source, *target) == -1) {
           error = "rename";
           return;
           }
        // delete all empty source directories
        recname = source;
        recname.Truncate(strlen(*recname) - 1);
        recname = StripLastDirectory(*recname);
        do {
           if (!RemoveEmptyDirectories(*recname, true))
              break;
           recname = StripLastDirectory(*recname);
           }
        while (strcmp(*recname, VideoDirectory));
        }
     else {
        int required = DirSizeMB(*source);
        int available = FreeDiskSpaceMB(*target);

        // validate free space
        if (required < available) {
           cReadDir d(*source);
           struct dirent *e;
           bool success = true;

           // allocate copying buffer
           const int len = 1024 * 1024;
           char *buffer = MALLOC(char, len);
           if (!buffer) {
              error = "MALLOC";
              return;
              }

           // loop through all files, but skip all sub-directories
           while (Running() && (e = d.Next()) != NULL) {
                 // skip generic entries
                 if (strcmp(e->d_name, ".") && strcmp(e->d_name, "..") && strcmp(e->d_name, "lost+found")) {
                    cString sourceFile = cString::sprintf("%s%s", *source, e->d_name);
                    cString targetFile = cString::sprintf("%s%s", *target, e->d_name);

                    // copy only regular files
                    struct stat sts;
                    if (!stat(*sourceFile, &sts) && S_ISREG(sts.st_mode)) {
                       int r = -1, w = -1;
                       cUnbufferedFile *inputFile = cUnbufferedFile::Create(*sourceFile, O_RDONLY | O_LARGEFILE);
                       cUnbufferedFile *outputFile = cUnbufferedFile::Create(*targetFile, O_RDWR | O_CREAT | O_LARGEFILE);

                       // validate files
                       if (!inputFile || !outputFile) {
                          success = false;
                          break;
                          }

                       // do actual copy
                       do {
                         r = inputFile->Read(buffer, len);
                         if (r > 0)
                            w = outputFile->Write(buffer, r);
                         else
                            w = 0;
                       } while (Running() && r > 0 && w > 0);
                       DELETENULL(inputFile);
                       DELETENULL(outputFile);

                       // validate result
                       if (!Running() || r < 0 || w < 0) {
                          success = false;
                          break;
                          }
                       }
                   }
                }

           // release allocated buffer
           free(buffer);

           // delete all created target files and directories
           if (!success) {
              target = StripLastDirectory(*target);
              RemoveFileOrDir(*target, true);
              target = StripLastDirectory(*target);
              RemoveEmptyDirectories(*target, true);
              error = "copy failed";
              return;
              }
           }
        else {
           // delete all created empty target directories
           recname = target;
           recname.Truncate(strlen(*recname) - 1);
           recname = StripLastDirectory(*recname);
           do {
              if (!RemoveEmptyDirectories(*recname, true))
                 break;
              recname = StripLastDirectory(*recname);
              }
           while (strcmp(*recname, VideoDirectory));
           error = "insufficient free space";
           return;
           }
        }

     if (deleteSource) {
        // Recordings' methods require the last directory delimiter to be stripped off
        source.Truncate(strlen(*source) - 1);
        cRecording *recording = Recordings.GetByName(*source);
        if (recording->Delete())
           Recordings.DelByName(*source, false);
        }
     else
        Recordings.TouchUpdate();
     }
}

// --- cFileTransfer ----------------------------------------------------------------

cMutex cFileTransfer::mutex;
char *cFileTransfer::copiedVersionName = NULL;
cCopyingThread *cFileTransfer::copyingThread = NULL;
bool cFileTransfer::error = false;
bool cFileTransfer::ended = false;

bool cFileTransfer::Start(cRecording *Recording, const char *FileName, bool CopyOnly)
{
  cMutexLock MutexLock(&mutex);
  if (!copyingThread) {
     cString NewName = NewVideoFileName(Recording->FileName(), FileName);
     error = false;
     ended = false;
     if (strlen(*NewName)) {
        copiedVersionName = strdup(*NewName);
        copyingThread = new cCopyingThread(Recording->FileName(), copiedVersionName, !CopyOnly);
        return true;
        }
     }
  return false;
}

void cFileTransfer::Stop(void)
{
  cMutexLock MutexLock(&mutex);
  bool Interrupted = copyingThread && copyingThread->Active();
  const char *Error = copyingThread ? copyingThread->Error() : NULL;
  DELETENULL(copyingThread);
  if (Interrupted || Error) {
     if (Interrupted)
        isyslog("file transfer has been interrupted");
     if (Error)
        esyslog("ERROR: '%s' during file transfer", Error);
     RemoveVideoFile(copiedVersionName); //XXX what if this file is currently being replayed?
     Recordings.DelByName(copiedVersionName);
     free(copiedVersionName);
     copiedVersionName = NULL;
     }
}

bool cFileTransfer::Active(void)
{
  cMutexLock MutexLock(&mutex);
  if (copyingThread) {
     if (copyingThread->Active())
        return true;
     error = copyingThread->Error();
     Stop();
     free(copiedVersionName);
     copiedVersionName = NULL;
     ended = true;
     }
  return false;
}

bool cFileTransfer::Error(void)
{
  cMutexLock MutexLock(&mutex);
  bool result = error;
  error = false;
  return result;
}

bool cFileTransfer::Ended(void)
{
  cMutexLock MutexLock(&mutex);
  bool result = ended;
  ended = false;
  return result;
}
