/*
 * freeboxRTSPClient.c: RTSP Client for the VDR FreeboxTv plugin
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id: freeboxRTSPClient.c 10 2007-09-11 16:14:11Z odj $
 */

// VDR includes
#include <vdr/device.h>
#include <vdr/thread.h>
#include <vdr/ringbuffer.h>

// LiveMedia includes
#include "liveMedia.hh"
#include "BasicUsageEnvironment.hh"
#include "GroupsockHelper.hh"
#include "RTSPClient.hh"

#include "freeboxtv.h"

void shutdown (int exitCode = 1);
void tearDownStreams ();

int verbosityLevel = 1;

UsageEnvironment *env;
RTSPClient *FreeboxClient = NULL;
MediaSession *Freeboxsession = NULL;
MediaSubsession* subsession;
void subsessionAfterPlaying(void* clientData);
extern unsigned short ClientPortNum;   // Port client  utiliser : 0 = random value
cRingBufferLinear *TSDataBuffer;

const char *plugName = "FreeboxTv VDR Plugin";
int rtspStreamOverTCP = 0;
char *url = new char[255];

using namespace FreeboxTv;
using FreeboxTv::cFreeboxClient;
cMutex fbmutex;

extern cFreeboxClient *freeboxClient;

// --- VdrSink --------------------------------------------------------
class VdrSink: public MediaSink {
public:
  static VdrSink* createNew(UsageEnvironment& env, unsigned bufferSize = 20000);

protected:
  VdrSink(UsageEnvironment& env, unsigned bufferSize);
  virtual ~VdrSink();

private:
  virtual Boolean continuePlaying();
  static void afterGettingFrame(void* clientData, unsigned frameSize,
				unsigned numTruncatedBytes,
				struct timeval presentationTime,
				unsigned durationInMicroseconds);
  void afterGettingFrame1(unsigned frameSize, struct timeval presentationTime);

  static void ourOnSourceClosure(void* clientData);

  unsigned char* fBuffer;
  unsigned fBufferSize;

};

VdrSink::VdrSink(UsageEnvironment& env, unsigned bufferSize) 
  : MediaSink(env), fBufferSize(bufferSize) {
  fBuffer = new unsigned char[bufferSize];
}

VdrSink::~VdrSink() {
  delete[] fBuffer;
}

VdrSink* VdrSink::createNew(UsageEnvironment& env, unsigned bufferSize) {
    return new VdrSink(env, bufferSize);
}

Boolean VdrSink::continuePlaying() {
  if (fSource == NULL) return False;

  fSource->getNextFrame(fBuffer, fBufferSize,
			afterGettingFrame, this,
			onSourceClosure, this);

  return True;
}

void VdrSink::afterGettingFrame(void* clientData, unsigned frameSize,
				 unsigned /*numTruncatedBytes*/,
				 struct timeval presentationTime,
				 unsigned /*durationInMicroseconds*/) {
  VdrSink* sink = (VdrSink*)clientData;
  sink->afterGettingFrame1(frameSize, presentationTime);
}

void VdrSink::afterGettingFrame1(unsigned frameSize,
				  struct timeval presentationTime) {
  fbmutex.Lock();
  int nbytes;
  if (TSDataBuffer)
    nbytes = TSDataBuffer->Put (fBuffer, frameSize);

  fbmutex.Unlock();

  // Then try getting the next frame:
  continuePlaying();
}

// --- cFreeboxClient --------------------------------------------------
cFreeboxClient::cFreeboxClient() {
  TaskScheduler* scheduler = BasicTaskScheduler::createNew();
  env = BasicUsageEnvironment::createNew(*scheduler);
  FreeboxClient = RTSPClient::createNew(*env, verbosityLevel, plugName);
  if (FreeboxClient == NULL) {
    *env << "Failed to create RTSP client: " << env->getResultMsg() << "\n";
    shutdown();
  }

  TSDataBuffer = new cRingBufferLinear (MEGABYTE(2), TS_SIZE);
  if (!TSDataBuffer)
    fprintf (stderr, "*** Echec creation TSDataBuffer\n");
}

cFreeboxClient::~cFreeboxClient() {
  delete url;
}

void cFreeboxClient::init_session (UsageEnvironment* env) {
  // Vrifie si une URL a t fournie (mais pas si elle est valide !!!)
  if (!strlen (url))
    return;

  // Ouvrir l'URL pour rcuprer la description SDP
  fprintf (stderr, "Url  lire : %s\n", url);
  char* sdpDescription = FreeboxClient->describeURL(url);
  if (sdpDescription == NULL) {
   *env << "Failed to get a SDP description from URL \"" << url
        << "\": " << env->getResultMsg() << "\n";
    shutdown();
  }

  *env << "Opened URL \"" << url
       << "\", returning a SDP description:\n" << sdpDescription << "\n";

  // Begin by sending an "OPTIONS" command:
  char* optionsResponse = FreeboxClient->sendOptionsCmd(url);
  *env << "Options retournes : " << optionsResponse << "\n";
   delete[] optionsResponse;

  // Create a media session object from this SDP description:
  Freeboxsession = MediaSession::createNew(*env, sdpDescription);
  delete[] sdpDescription;
  if (Freeboxsession == NULL) {
    *env << "Failed to create a MediaSession object from the SDP description: " << env->getResultMsg() << "\n";
    shutdown();
  }

  // Create RTP receivers (sources) for each subsession:
  MediaSubsessionIterator iter(*Freeboxsession);

  while ((subsession = iter.next()) != NULL) {
    // Set desired client port number if specified (0 = random value)
    if (ClientPortNum) {
      subsession->setClientPortNum(ClientPortNum);
    }

    if (!subsession->initiate()) {
      fprintf(stderr, "Failed to initiate \"%s/%s\" RTP subsession: %s\n", subsession->mediumName(), subsession->codecName(), env->getResultMsg());
    } else {
      fprintf(stderr, "Initiated \"%s/%s\" RTP subsession on port %d\n", subsession->mediumName(), subsession->codecName(), subsession->clientPortNum());

      // Issue a RTSP "SETUP" command on the chosen subsession:
      FreeboxClient->setupMediaSubsession(*subsession, False, rtspStreamOverTCP);

      if (strcmp(subsession->mediumName(), "video") == 0) {
        FreeboxClient->playMediaSubsession (*subsession);

        VdrSink* vdrSink;
        vdrSink = VdrSink::createNew(*env, MEGABYTE (2));
        subsession->sink = vdrSink;
        if (subsession->sink == NULL) {
          *env << "Failed to create VdrSink\"" 
               << "\": " << env->getResultMsg() << "\n";
        } else {
          *env << "Outputting data from the \"" << subsession->mediumName()
               << "/" << subsession->codecName()
               << "\" subsession to vdrSink\n";
        }

        subsession->sink->startPlaying(*(subsession->readSource()),
                                         subsessionAfterPlaying,
                                         subsession);
      }
    }
  }

  FreeboxClient->playMediaSession(*Freeboxsession);

  this->Start();    // Start taskScheduler loop in separate a thread
}

void cFreeboxClient::Action () {
  // Appelle la boucle sans fin "env->taskScheduler().doEventLoop()"
  env->taskScheduler().doEventLoop(); // does not return
}


void cFreeboxClient::shutdown() {
  // Teardown, then shutdown, any outstanding RTP/RTCP subsessions
  FreeboxClient->teardownMediaSession(*Freeboxsession);
  RTSPClient::close(Freeboxsession);

  // Finally, shut down our client:
  RTSPClient::close(FreeboxClient);

  // Stop taskScheduler loop thread
  this->Cancel();
}

void subsessionAfterPlaying(void* clientData) {
  // Begin by closing this media subsession:
  MediaSubsession* subsession = (MediaSubsession*)clientData;
  Medium::close(subsession->sink);
  subsession->sink = NULL;

  // Next, check whether *all* subsessions have now been closed:
  MediaSession& session = subsession->parentSession();
  MediaSubsessionIterator iter(session);
  while ((subsession = iter.next()) != NULL) {
    if (subsession->sink != NULL) return; // this subsession is still active
  }
}

