/*
 * Softcam plugin to VDR (C++)
 *
 * This code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "nagra2.h"

// -- cAuxSrv ------------------------------------------------------------------

#ifdef HAS_AUXSRV
#include "network.h"
#define AUX_PROTOCOL_VERSION 2
int auxEnabled=0;
int auxPort=7777;
char auxAddr[80]="localhost";
char auxPassword[250]="auxserver";

class cAuxSrv : public cMutex {
private:
  cNetSocket so;
  //
  bool Login(void);
public:
  cAuxSrv(void);
  int Map(int map, unsigned char *data, int len, int outlen);
  };

cAuxSrv::cAuxSrv(void)
:so(DEFAULT_CONNECT_TIMEOUT,7,DEFAULT_IDLE_TIMEOUT)
{}

bool cAuxSrv::Login()
{
  unsigned char buff[256];
  PRINTF(L_SYS_MAP,"auxsrv: connecting to %s:%d",auxAddr,auxPort);
  if(so.Connect(auxAddr,auxPort)) {
    buff[0]=0xA7;
    buff[1]=0x7A;
    buff[2]=0;
    int l=strlen(auxPassword);
    buff[3]=l;
    memcpy(&buff[4],auxPassword,l);
    buff[4+l]=0xFF;
    if(so.Write(buff,l+5)==l+5 &&
       so.Read(buff,sizeof(buff))>=9 &&
       buff[0]==0x7A && buff[1]==0xA7 && buff[2]==0x00 && buff[3]==0x04 && buff[8]==0xff &&
       ((buff[4]<<8)|buff[5])==AUX_PROTOCOL_VERSION) return true;
    PRINTF(L_SYS_MAP,"auxsrv: login write failed");
    }
  so.Disconnect();
  return false;
}

int cAuxSrv::Map(int map, unsigned char *data, int len, int outlen)
{
  if(!auxEnabled) {
    PRINTF(L_SYS_MAP,"auxsrv: AUXserver is disabled!");
    return -1;
    }
  if(len>500 || outlen>500) return -1;
  cMutexLock lock(this);
  if(!so.Connected() && !Login()) return false;
  PRINTF(L_SYS_MAP,"auxsrv: calling map%02x",map);
  unsigned char buff[512];
  buff[0]=0xA7;
  buff[1]=0x7A;
  buff[2]=((len+1)>>8) & 0xff;
  buff[3]=(len+1)&0xff;
  buff[4]=map;
  memcpy(&buff[5],data,len);
  buff[len+5]=0xFF;
  if(so.Write(buff,len+6)==len+6) {
    if((len=so.Read(buff,sizeof(buff)))>0) {
      if(buff[0]==0x7A && buff[1]==0xA7) {
        if(buff[4]==0x00) {
          int cycles=(buff[5]<<16)|(buff[6]<<8)|buff[7];
          int l=buff[2]*256+buff[3];
          if(len>=l+5 && l==outlen+4) {
            if(buff[l+4]==0xFF) {
              memcpy(data,buff+8,outlen);
              return cycles;
              }
            else PRINTF(L_SYS_MAP,"auxsrv: bad footer in map%02x response",map);
            }
          else PRINTF(L_SYS_MAP,"auxsrv: bad length in map%02x response (got=%d,%d want=%d,%d)",map,l-4,len,outlen,l+8);
          }
        else PRINTF(L_SYS_MAP,"auxsrv: map%02x not successfull (unsupported?)",map);
        }
      else PRINTF(L_SYS_MAP,"auxsrv: bad response to map%02x",map);
      }
    else PRINTF(L_SYS_MAP,"auxsrv: map%02x read failed",map);
    }
  else  PRINTF(L_SYS_MAP,"auxsrv: map%02x write failed",map);
  so.Disconnect();
  return -1;
}
#endif //HAS_AUXSRV

// -- cMap0101 ----------------------------------------------------------------

class cMap0101 : public cMapCore {
private:
  static const unsigned char primes[];
  static const unsigned short coef22[][32];
#ifdef HAS_AUXSRV
  cAuxSrv aux;
#endif
  //
  void MakePrime(BIGNUM *n, unsigned char *residues);
protected:
  void DoMap(int f, unsigned char *data=0, int l=0);
  };

const unsigned char cMap0101::primes[] = {
  0x03,0x05,0x07,0x0B,0x0D,0x11,0x13,0x17,0x1D,0x1F,0x25,0x29,0x2B,0x2F,0x35,0x3B,
  0x3D,0x43,0x47,0x49,0x4F,0x53,0x59,0x61,0x65,0x67,0x6B,0x6D,0x71,0x7F,0x83,0x89,
  0x8B,0x95,0x97,0x9D,0xA3,0xA7,0xAD,0xB3,0xB5,0xBF,0xC1,0xC5,0xC7,0xD3,0xDF,0xE3,
  0xE5,0xE9,0xEF,0xF1,0xFB
  };

const unsigned short cMap0101::coef22[256][32] = {
  {0,1,1,0,1,0,1,1,0,11,10,11,10,10,11,10,11,21,22,22,21,22,21,22,22,32,33,32,33,33,32,33},
  {3,1,0,1,1,0,1,0,1,10,11,11,10,11,10,11,11,21,22,21,22,22,21,22,21,33,33,33,33,32,33,33},
  {5,1,0,1,1,0,1,0,1,10,11,11,10,11,10,11,11,21,22,21,22,22,21,22,21,33,33,33,33,32,33,33},
  {7,1,1,1,0,1,1,1,1,10,11,10,11,11,10,11,10,22,22,22,22,21,22,22,22,33,32,33,33,33,33,32},
  {9,1,1,1,0,1,1,1,1,10,11,10,11,11,10,11,10,22,22,22,22,21,22,22,22,33,32,33,33,33,33,32},
  {12,0,1,1,1,1,0,1,1,11,11,11,11,10,11,11,11,22,21,22,22,22,22,21,22,33,33,33,34,33,33,33},
  {14,0,1,1,1,1,0,1,1,11,11,11,11,10,11,11,11,22,21,22,22,22,22,21,22,33,33,33,34,33,33,33},
  {16,1,1,2,1,1,1,1,2,11,10,11,11,11,11,10,11,22,22,22,23,22,22,22,22,34,33,33,33,33,34,33},
  {0,1,1,2,1,1,1,1,2,11,10,11,11,11,11,10,11,22,22,22,23,22,22,22,22,34,33,33,33,33,34,33},
  {3,1,1,1,1,2,1,1,1,11,11,11,12,11,11,11,11,23,22,22,22,22,23,22,22,33,33,34,33,34,33,33},
  {5,1,1,1,1,2,1,1,1,11,11,11,12,11,11,11,11,23,22,22,22,22,23,22,22,33,33,34,33,34,33,33},
  {7,1,2,1,2,1,1,2,1,12,11,11,11,11,12,11,11,22,22,23,22,23,22,22,23,33,34,33,33,34,33,34},
  {9,1,2,1,2,1,1,2,1,12,11,11,11,11,12,11,11,22,22,23,22,23,22,22,23,33,34,33,33,34,33,34},
  {12,2,1,1,2,1,2,1,1,11,11,12,11,12,11,11,12,22,23,22,22,23,22,23,22,34,34,33,34,33,34,34},
  {14,2,1,1,2,1,2,1,1,11,11,12,11,12,11,11,12,22,23,22,22,23,22,23,22,34,34,33,34,33,34,34},
  {16,2,1,2,1,2,2,1,2,11,12,11,11,12,11,12,11,23,23,22,23,22,23,23,22,34,33,34,34,33,34,33},
  {0,2,1,2,1,2,2,1,2,11,12,11,11,12,11,12,11,23,23,22,23,22,23,23,22,34,33,34,34,33,34,33},
  {3,1,2,2,1,2,1,2,2,12,12,11,12,11,12,12,11,23,22,23,23,22,23,22,23,34,34,34,33,34,34,34},
  {5,1,2,2,1,2,1,2,2,12,12,11,12,11,12,12,11,23,22,23,23,22,23,22,23,34,34,34,33,34,34,34},
  {7,2,2,1,2,2,2,2,1,12,11,12,12,11,12,11,12,23,23,23,22,23,23,23,23,33,34,34,34,34,33,34},
  {9,2,2,1,2,2,2,2,1,12,11,12,12,11,12,11,12,23,23,23,22,23,23,23,23,33,34,34,34,34,33,34},
  {12,2,2,2,2,1,2,2,2,12,12,12,11,12,12,12,12,22,23,23,23,23,22,23,23,34,34,35,34,34,34,34},
  {14,2,2,2,2,1,2,2,2,12,12,12,11,12,12,12,12,22,23,23,23,23,22,23,23,34,34,35,34,34,34,34},
  {16,2,3,2,2,2,2,3,2,11,12,12,12,12,11,12,12,23,23,24,23,23,23,23,24,34,34,34,34,35,34,34},
  {0,2,3,2,2,2,2,3,2,11,12,12,12,12,11,12,12,23,23,24,23,23,23,23,24,34,34,34,34,35,34,34},
  {3,2,2,2,3,2,2,2,2,12,12,13,12,12,12,12,13,23,23,23,23,24,23,23,23,34,35,34,35,34,34,35},
  {5,2,2,2,3,2,2,2,2,12,12,13,12,12,12,12,13,23,23,23,23,24,23,23,23,34,35,34,35,34,34,35},
  {7,3,2,3,2,2,3,2,3,12,12,12,12,13,12,12,12,23,24,23,24,23,23,24,23,35,34,34,35,34,35,34},
  {9,3,2,3,2,2,3,2,3,12,12,12,12,13,12,12,12,23,24,23,24,23,23,24,23,35,34,34,35,34,35,34},
  {12,2,2,3,2,3,2,2,3,12,13,12,13,12,12,13,12,24,23,23,24,23,24,23,23,35,34,35,34,35,35,34},
  {14,2,2,3,2,3,2,2,3,12,13,12,13,12,12,13,12,24,23,23,24,23,24,23,23,35,34,35,34,35,35,34},
  {16,2,3,2,3,3,2,3,2,13,12,12,13,12,13,12,12,24,23,24,23,24,24,23,24,34,35,35,34,35,34,35},
  {0,2,3,2,3,3,2,3,2,13,12,12,13,12,13,12,12,24,23,24,23,24,24,23,24,34,35,35,34,35,34,35},
  {3,3,3,2,3,2,3,3,2,13,12,13,12,13,13,12,13,23,24,24,23,24,23,24,24,35,35,34,35,35,35,35},
  {5,3,3,2,3,2,3,3,2,13,12,13,12,13,13,12,13,23,24,24,23,24,23,24,24,35,35,34,35,35,35,35},
  {7,3,2,3,3,3,3,2,3,12,13,13,12,13,12,13,13,24,24,23,24,24,24,24,23,35,35,35,35,34,35,35},
  {9,3,2,3,3,3,3,2,3,12,13,13,12,13,12,13,13,24,24,23,24,24,24,24,23,35,35,35,35,34,35,35},
  {12,3,3,3,2,3,3,3,3,13,13,12,13,13,13,13,12,24,24,24,24,23,24,24,24,35,36,35,35,35,35,36},
  {14,3,3,3,2,3,3,3,3,13,13,12,13,13,13,13,12,24,24,24,24,23,24,24,24,35,36,35,35,35,35,36},
  {16,4,3,3,3,3,4,3,3,13,13,13,13,12,13,13,13,24,25,24,24,24,24,25,24,35,35,35,36,35,35,35},
  {0,4,3,3,3,3,4,3,3,13,13,13,13,12,13,13,13,24,25,24,24,24,24,25,24,35,35,35,36,35,35,35},
  {3,3,3,4,3,3,3,3,4,13,14,13,13,13,13,14,13,24,24,24,25,24,24,24,24,36,35,36,35,35,36,35},
  {5,3,3,4,3,3,3,3,4,13,14,13,13,13,13,14,13,24,24,24,25,24,24,24,24,36,35,36,35,35,36,35},
  {7,3,4,3,3,4,3,4,3,13,13,13,14,13,13,13,13,25,24,25,24,24,25,24,25,35,35,36,35,36,35,35},
  {9,3,4,3,3,4,3,4,3,13,13,13,14,13,13,13,13,25,24,25,24,24,25,24,25,35,35,36,35,36,35,35},
  {12,3,4,3,4,3,3,4,3,14,13,14,13,13,14,13,14,24,24,25,24,25,24,24,25,35,36,35,36,36,35,36},
  {14,3,4,3,4,3,3,4,3,14,13,14,13,13,14,13,14,24,24,25,24,25,24,24,25,35,36,35,36,36,35,36},
  {16,4,3,4,4,3,4,3,4,13,13,14,13,14,13,13,14,24,25,24,25,25,24,25,24,36,36,35,36,35,36,36},
  {0,4,3,4,4,3,4,3,4,13,13,14,13,14,13,13,14,24,25,24,25,25,24,25,24,36,36,35,36,35,36,36},
  {3,4,3,4,3,4,4,3,4,13,14,13,14,14,13,14,13,25,25,24,25,24,25,25,24,36,35,36,36,36,36,35},
  {5,4,3,4,3,4,4,3,4,13,14,13,14,14,13,14,13,25,25,24,25,24,25,25,24,36,35,36,36,36,36,35},
  {7,3,4,4,4,4,3,4,4,14,14,13,14,13,14,14,13,25,24,25,25,25,25,24,25,36,36,36,35,36,36,36},
  {9,3,4,4,4,4,3,4,4,14,14,13,14,13,14,14,13,25,24,25,25,25,25,24,25,36,36,36,35,36,36,36},
  {12,4,4,3,4,4,4,4,3,14,13,14,14,14,14,13,14,25,25,25,24,25,25,25,25,37,36,36,36,36,37,36},
  {14,4,4,3,4,4,4,4,3,14,13,14,14,14,14,13,14,25,25,25,24,25,25,25,25,37,36,36,36,36,37,36},
  {16,4,4,4,4,5,4,4,4,14,14,14,13,14,14,14,14,26,25,25,25,25,26,25,25,36,36,37,36,36,36,36},
  {0,4,4,4,4,5,4,4,4,14,14,14,13,14,14,14,14,26,25,25,25,25,26,25,25,36,36,37,36,36,36,36},
  {3,4,5,4,4,4,4,5,4,15,14,14,14,14,15,14,14,25,25,26,25,25,25,25,26,36,37,36,36,37,36,37},
  {5,4,5,4,4,4,4,5,4,15,14,14,14,14,15,14,14,25,25,26,25,25,25,25,37,36,37,36,36,37,36,37},
  {7,5,4,4,5,4,5,4,4,14,14,15,14,14,14,14,15,25,26,25,25,26,25,26,36,36,37,36,37,36,36,37},
  {9,5,4,4,5,4,5,4,4,14,14,15,14,14,14,14,26,25,26,25,25,26,25,26,36,36,37,36,37,36,36,37},
  {12,5,4,5,4,4,5,4,5,14,15,14,14,15,14,15,25,25,26,25,26,25,25,26,36,37,36,37,37,36,37,36},
  {14,5,4,5,4,4,5,4,15,14,15,14,14,15,14,15,25,25,26,25,26,25,25,26,36,37,36,37,37,36,37,36},
  {16,4,5,5,4,5,4,5,14,14,15,14,15,14,14,15,25,26,25,26,26,25,26,25,37,37,36,37,36,37,37,36},
  {40,4,5,5,4,5,4,5,14,14,15,14,15,14,14,15,25,26,25,26,26,25,26,25,37,37,36,37,36,37,37,36},
  {44,4,5,4,5,5,4,5,14,15,14,15,15,14,15,14,26,26,25,26,25,26,26,25,37,36,37,37,37,37,36,37},
  {44,4,5,4,5,5,4,5,14,15,14,15,15,14,15,14,26,26,25,26,25,26,26,25,37,36,37,37,37,37,36,37},
  {47,5,5,5,5,4,5,5,15,15,14,15,14,15,15,14,26,25,26,26,26,26,25,26,37,37,37,36,37,37,37,37},
  {47,5,5,5,5,4,5,5,15,15,14,15,14,15,15,14,26,25,26,26,26,26,25,26,37,37,37,36,37,37,37,37},
  {52,5,4,5,5,5,5,4,15,14,15,15,15,15,14,15,26,26,26,25,26,26,26,26,38,37,37,37,37,38,37,37},
  {52,5,4,5,5,5,5,4,15,14,15,15,15,15,14,15,26,26,26,25,26,26,26,26,38,37,37,37,37,38,37,37},
  {56,5,5,5,6,5,5,5,15,15,15,14,15,15,15,15,27,26,26,26,26,27,26,26,37,37,38,37,37,37,37,38},
  {56,5,5,5,6,5,5,5,15,15,15,14,15,15,15,15,27,26,26,26,26,27,26,26,37,37,38,37,37,37,37,38},
  {60,6,5,5,5,5,6,5,16,15,15,15,15,16,15,15,26,26,27,26,26,26,26,27,37,38,37,37,38,37,38,37},
  {60,6,5,5,5,5,6,5,16,15,15,15,15,16,15,15,26,26,27,26,26,26,26,27,37,38,37,37,38,37,38,37},
  {65,5,5,6,5,6,5,5,15,15,16,15,15,15,15,16,26,27,26,26,27,26,27,26,37,38,37,38,37,37,38,37},
  {65,5,5,6,5,6,5,5,15,15,16,15,15,15,15,16,26,27,26,26,27,26,27,26,37,38,37,38,37,37,38,37},
  {69,5,6,5,5,6,5,6,15,16,15,15,16,15,16,15,26,27,26,27,26,26,27,26,38,37,38,38,37,38,37,38},
  {69,5,6,5,5,6,5,6,15,16,15,15,16,15,16,15,26,27,26,27,26,26,27,26,38,37,38,38,37,38,37,38},
  {72,6,6,5,6,5,6,6,15,16,15,16,15,15,16,15,27,26,27,27,26,27,26,27,38,37,38,37,38,38,37,38},
  {72,6,6,5,6,5,6,6,15,16,15,16,15,15,16,15,27,26,27,27,26,27,26,27,38,37,38,37,38,38,37,38},
  {76,6,5,6,6,5,6,5,16,15,16,16,15,16,15,16,27,26,27,26,27,27,26,27,37,38,38,38,38,37,38,38},
  {76,6,5,6,6,5,6,5,16,15,16,16,15,16,15,16,27,26,27,26,27,27,26,27,37,38,38,38,38,37,38,38},
  {81,6,6,6,5,6,6,6,16,15,16,15,16,16,15,16,26,27,27,27,27,26,27,27,38,38,37,38,38,38,38,37},
  {81,6,6,6,5,6,6,6,16,15,16,15,16,16,15,16,26,27,27,27,27,26,27,27,38,38,37,38,38,38,38,37},
  {85,5,6,6,6,6,5,6,15,16,16,16,16,15,16,16,27,27,26,27,27,27,27,26,38,38,38,38,39,38,38,38},
  {85,5,6,6,6,6,5,6,15,16,16,16,16,15,16,16,27,27,26,27,27,27,27,26,38,38,38,38,39,38,38,38},
  {89,6,6,7,6,6,6,6,16,16,15,16,16,16,16,15,27,27,27,27,28,27,27,27,38,39,38,38,38,38,39,38},
  {89,6,6,7,6,6,6,6,16,16,15,16,16,16,16,15,27,27,27,27,28,27,27,27,38,39,38,38,38,38,39,38},
  {94,6,6,6,6,7,6,6,16,16,16,16,17,16,16,16,27,28,27,27,27,27,28,27,39,38,38,39,38,39,38,38},
  {94,6,6,6,6,7,6,6,16,16,16,16,17,16,16,16,27,28,27,27,27,27,28,27,39,38,38,39,38,39,38,38},
  {97,6,7,6,7,6,6,7,16,17,16,16,16,16,17,16,28,27,27,28,27,28,27,27,39,38,39,38,38,39,38,39},
  {97,6,7,6,7,6,6,7,16,17,16,16,16,16,17,16,28,27,27,28,27,28,27,27,39,38,39,38,38,39,38,39},
  {101,7,6,6,7,6,7,6,17,16,16,17,16,17,16,16,28,27,28,27,27,28,27,28,38,39,39,38,39,38,39,39},
  {101,7,6,6,7,6,7,6,17,16,16,17,16,17,16,16,28,27,28,27,27,28,27,28,38,39,39,38,39,38,39,39},
  {106,7,6,7,6,7,7,6,17,16,17,16,16,17,16,17,27,28,28,27,28,27,28,28,38,39,38,39,39,38,39,38},
  {106,7,6,7,6,7,7,6,17,16,17,16,16,17,16,17,27,28,28,27,28,27,28,28,38,39,38,39,39,38,39,38},
  {110,6,7,7,6,7,6,7,16,17,17,16,17,16,17,17,27,28,27,28,28,27,28,27,39,39,39,39,38,39,39,39},
  {110,6,7,7,6,7,6,7,16,17,17,16,17,16,17,17,27,28,27,28,28,27,28,27,39,39,39,39,38,39,39,39},
  {114,7,7,6,7,7,7,7,16,17,16,17,17,16,17,16,28,28,28,28,27,28,28,28,39,38,39,39,39,39,38,39},
  {114,7,7,6,7,7,7,7,16,17,16,17,17,16,17,16,28,28,28,28,27,28,28,28,39,38,39,39,39,39,38,39},
  {117,7,7,7,7,6,7,7,17,17,17,17,16,17,17,17,28,27,28,28,28,28,27,28,39,39,39,40,39,39,39,39},
  {117,7,7,7,7,6,7,7,17,17,17,17,16,17,17,17,28,27,28,28,28,28,27,28,39,39,39,40,39,39,39,39},
  {122,7,8,7,7,7,7,8,17,16,17,17,17,17,16,17,28,28,28,29,28,28,28,28,40,39,39,39,39,40,39,39},
  {122,7,8,7,7,7,7,8,17,16,17,17,17,17,16,17,28,28,28,29,28,28,28,28,40,39,39,39,39,40,39,39},
  {126,7,7,7,8,7,7,7,17,17,17,18,17,17,17,17,29,28,28,28,28,29,28,28,39,39,40,39,40,39,39,40},
  {126,7,7,7,8,7,7,7,17,17,17,18,17,17,17,17,29,28,28,28,28,29,28,28,39,39,40,39,40,39,39,40},
  {130,8,7,8,7,7,8,7,18,17,17,17,17,18,17,17,28,28,29,28,29,28,28,29,39,40,39,39,40,39,40,39},
  {130,8,7,8,7,7,8,7,18,17,17,17,17,18,17,17,28,28,29,28,29,28,28,29,39,40,39,39,40,39,40,39},
  {135,7,7,8,7,8,7,7,17,17,18,17,18,17,17,18,28,29,28,28,29,28,29,28,40,40,39,40,39,40,40,39},
  {135,7,7,8,7,8,7,7,17,17,18,17,18,17,17,18,28,29,28,28,29,28,29,28,40,40,39,40,39,40,40,39},
  {139,7,8,7,8,8,7,8,17,18,17,17,18,17,18,17,29,29,28,29,28,29,29,28,40,39,40,40,39,40,39,40},
  {139,7,8,7,8,8,7,8,17,18,17,17,18,17,18,17,29,29,28,29,28,29,29,28,40,39,40,40,39,40,39,40},
  {142,8,8,7,8,7,8,8,18,18,17,18,17,18,18,17,29,28,29,29,28,29,28,29,40,40,40,39,40,40,40,40},
  {142,8,8,7,8,7,8,8,18,18,17,18,17,18,18,17,29,28,29,29,28,29,28,29,40,40,40,39,40,40,40,40},
  {147,8,7,8,8,8,8,7,18,17,18,18,17,18,17,18,29,29,29,28,29,29,29,29,39,40,40,40,40,39,40,40},
  {147,8,7,8,8,8,8,7,18,17,18,18,17,18,17,18,29,29,29,28,29,29,29,29,39,40,40,40,40,39,40,40},
  {151,8,8,8,7,8,8,8,18,18,18,17,18,18,18,18,28,29,29,29,29,28,29,29,40,40,41,40,40,40,40,41},
  {151,8,8,8,7,8,8,8,18,18,18,17,18,18,18,18,28,29,29,29,29,28,29,29,40,40,41,40,40,40,40,41},
  {155,9,8,8,8,8,9,8,17,18,18,18,18,17,18,18,29,29,30,29,29,29,29,30,40,40,40,40,41,40,40,40},
  {155,9,8,8,8,8,9,8,17,18,18,18,18,17,18,18,29,29,30,29,29,29,29,30,40,40,40,40,41,40,40,40},
  {159,8,8,9,8,8,8,8,18,18,19,18,18,18,18,19,29,29,29,29,30,29,29,29,40,41,40,41,40,40,41,40},
  {159,8,8,9,8,8,8,8,18,18,19,18,18,18,18,19,29,29,29,29,30,29,29,29,40,41,40,41,40,40,41,40},
  {164,8,9,8,8,9,8,9,18,18,18,18,19,18,18,18,29,30,29,30,29,29,30,29,41,40,40,41,40,41,40,40},
  {164,8,9,8,8,9,8,9,18,18,18,18,19,18,18,18,29,30,29,30,29,29,30,29,41,40,40,41,40,41,40,40},
  {167,8,9,8,9,8,8,9,18,19,18,19,18,18,19,18,30,29,29,30,29,30,29,29,41,40,41,40,41,41,40,41},
  {167,8,9,8,9,8,8,9,18,19,18,19,18,18,19,18,30,29,29,30,29,30,29,29,41,40,41,40,41,41,40,41},
  {171,9,8,9,9,8,9,8,19,18,18,19,18,19,18,18,30,29,30,29,30,30,29,30,40,41,41,40,41,40,41,41},
  {171,9,8,9,9,8,9,8,19,18,18,19,18,19,18,18,30,29,30,29,30,30,29,30,40,41,41,40,41,40,41,41},
  {176,9,8,9,8,9,9,8,19,18,19,18,19,19,18,19,29,30,30,29,30,29,30,30,41,41,40,41,41,41,41,40},
  {176,9,8,9,8,9,9,8,19,18,19,18,19,19,18,19,29,30,30,29,30,29,30,30,41,41,40,41,41,41,41,40},
  {180,8,9,9,9,9,8,9,18,19,19,18,19,18,19,19,30,30,29,30,30,30,30,29,41,41,41,41,40,41,41,41},
  {180,8,9,9,9,9,8,9,18,19,19,18,19,18,19,19,30,30,29,30,30,30,30,29,41,41,41,41,40,41,41,41},
  {184,9,9,8,9,9,9,9,19,19,18,19,19,19,19,18,30,30,30,30,29,30,30,30,41,42,41,41,41,41,42,41},
  {184,9,9,8,9,9,9,9,19,19,18,19,19,19,19,18,30,30,30,30,29,30,30,30,41,42,41,41,41,41,42,41},
  {189,9,9,9,9,10,9,9,19,19,19,19,18,19,19,19,30,31,30,30,30,30,31,30,41,41,41,42,41,41,41,41},
  {189,9,9,9,9,10,9,9,19,19,19,19,18,19,19,19,30,31,30,30,30,30,31,30,41,41,41,42,41,41,41,41},
  {192,9,10,9,9,9,9,10,19,20,19,19,19,19,20,19,30,30,30,31,30,30,30,30,42,41,42,41,41,42,41,42},
  {192,9,10,9,9,9,9,10,19,20,19,19,19,19,20,19,30,30,30,31,30,30,30,30,42,41,42,41,41,42,41,42},
  {196,10,9,9,10,9,10,9,19,19,19,20,19,19,19,19,31,30,31,30,30,31,30,31,41,41,42,41,42,41,41,42},
  {196,10,9,9,10,9,10,9,19,19,19,20,19,19,19,19,31,30,31,30,30,31,30,31,41,41,42,41,42,41,41,42},
  {200,10,9,10,9,9,10,9,20,19,20,19,19,20,19,20,30,30,31,30,31,30,30,31,41,42,41,42,42,41,42,41},
  {200,10,9,10,9,9,10,9,20,19,20,19,19,20,19,20,30,30,31,30,31,30,30,31,41,42,41,42,42,41,42,41},
  {205,9,10,10,9,10,9,10,19,19,20,19,20,19,19,20,30,31,30,31,31,30,31,30,42,42,41,42,41,42,42,41},
  {205,9,10,10,9,10,9,10,19,19,20,19,20,19,19,20,30,31,30,31,31,30,31,30,42,42,41,42,41,42,42,41},
  {209,9,10,9,10,10,9,10,19,20,19,20,20,19,20,19,31,31,30,31,30,31,31,30,42,41,42,42,42,42,41,42},
  {209,9,10,9,10,10,9,10,19,20,19,20,20,19,20,19,31,31,30,31,30,31,31,30,42,41,42,42,42,42,41,42},
  {212,10,10,10,10,9,10,10,20,20,19,20,19,20,20,19,31,30,31,31,31,31,30,31,42,42,42,41,42,42,42,42},
  {212,10,10,10,10,9,10,10,20,20,19,20,19,20,20,19,31,30,31,31,31,31,30,31,42,42,42,41,42,42,42,42},
  {217,10,9,10,10,10,10,9,20,19,20,20,20,20,19,20,31,31,31,30,31,31,31,31,43,42,42,42,42,43,42,42},
  {217,10,9,10,10,10,10,9,20,19,20,20,20,20,19,20,31,31,31,30,31,31,31,31,43,42,42,42,42,43,42,42},
  {221,10,10,10,11,10,10,10,20,20,20,19,20,20,20,20,32,31,31,31,31,32,31,31,42,42,43,42,42,42,42,43},
  {221,10,10,10,11,10,10,10,20,20,20,19,20,20,20,20,32,31,31,31,31,32,31,31,42,42,43,42,42,42,42,43},
  {225,11,10,10,10,10,11,10,21,20,20,20,20,21,20,20,31,31,32,31,31,31,31,32,42,43,42,42,43,42,43,42},
  {225,11,10,10,10,10,11,10,21,20,20,20,20,21,20,20,31,31,32,31,31,31,31,32,42,43,42,42,43,42,43,42},
  {230,10,10,11,10,11,10,10,20,20,21,20,20,20,20,21,31,32,31,31,32,31,32,31,42,43,42,43,42,42,43,42},
  {230,10,10,11,10,11,10,10,20,20,21,20,20,20,20,21,31,32,31,31,32,31,32,31,42,43,42,43,42,42,43,42},
  {234,10,11,10,10,11,10,11,20,21,20,20,21,20,21,20,31,32,31,32,31,31,32,31,43,42,43,43,42,43,42,43},
  {234,10,11,10,10,11,10,11,20,21,20,20,21,20,21,20,31,32,31,32,31,31,32,31,43,42,43,43,42,43,42,43},
  {237,11,11,10,11,10,11,11,20,21,20,21,20,20,21,20,32,31,32,32,31,32,31,32,43,42,43,42,43,43,42,43},
  {237,11,11,10,11,10,11,11,20,21,20,21,20,20,21,20,32,31,32,32,31,32,31,32,43,42,43,42,43,43,42,43},
  {241,11,10,11,11,10,11,10,21,20,21,21,20,21,20,21,32,31,32,31,32,32,31,32,42,43,43,43,43,42,43,43},
  {241,11,10,11,11,10,11,10,21,20,21,21,20,21,20,21,32,31,32,31,32,32,31,32,42,43,43,43,43,42,43,43},
  {246,11,11,11,10,11,11,11,21,20,21,20,21,21,20,21,31,32,32,32,32,31,32,32,43,43,42,43,43,43,43,42},
  {246,11,11,11,10,11,11,11,21,20,21,20,21,21,20,21,31,32,32,32,32,31,32,32,43,43,42,43,43,43,43,42},
  {250,10,11,11,11,11,10,11,20,21,21,21,21,20,21,21,32,32,31,32,32,32,32,31,43,43,43,43,44,43,43,43},
  {250,10,11,11,11,11,10,11,20,21,21,21,21,20,21,21,32,32,31,32,32,32,32,31,43,43,43,43,44,43,43,43},
  {254,11,11,12,11,11,11,11,21,21,20,21,21,21,21,20,32,32,32,32,33,32,32,32,43,44,43,43,43,43,44,43},
  {254,11,11,12,11,11,11,11,21,21,20,21,21,21,21,20,32,32,32,32,33,32,32,32,43,44,43,43,43,43,44,43},
  {259,11,11,11,11,12,11,11,21,21,21,21,22,21,21,21,32,33,32,32,32,32,33,32,44,43,43,44,43,44,43,43},
  {259,11,11,11,11,12,11,11,21,21,21,21,22,21,21,21,32,33,32,32,32,32,33,32,44,43,43,44,43,44,43,43},
  {262,11,12,11,12,11,11,12,21,22,21,21,21,21,22,21,33,32,32,33,32,33,32,32,44,43,44,43,43,44,43,44},
  {262,11,12,11,12,11,11,12,21,22,21,21,21,21,22,21,33,32,32,33,32,33,32,32,44,43,44,43,43,44,43,44},
  {266,12,11,11,12,11,12,11,22,21,21,22,21,22,21,21,33,32,33,32,32,33,32,33,43,44,44,43,44,43,44,44},
  {266,12,11,11,12,11,12,11,22,21,21,22,21,22,21,21,33,32,33,32,32,33,32,33,43,44,44,43,44,43,44,44},
  {271,12,11,12,11,12,12,11,22,21,22,21,21,22,21,22,32,33,33,32,33,32,33,33,43,44,43,44,44,43,44,43},
  {271,12,11,12,11,12,12,11,22,21,22,21,21,22,21,22,32,33,33,32,33,32,33,33,43,44,43,44,44,43,44,43},
  {275,11,12,12,11,12,11,12,21,22,22,21,22,21,22,22,32,33,32,33,33,32,33,32,44,44,44,44,43,44,44,44},
  {275,11,12,12,11,12,11,12,21,22,22,21,22,21,22,22,32,33,32,33,33,32,33,32,44,44,44,44,43,44,44,44},
  {279,12,12,11,12,12,12,12,21,22,21,22,22,21,22,21,33,33,33,33,32,33,33,33,44,43,44,44,44,44,43,44},
  {279,12,12,11,12,12,12,12,21,22,21,22,22,21,22,21,33,33,33,33,32,33,33,33,44,43,44,44,44,44,43,44},
  {282,12,12,12,12,11,12,12,22,22,22,22,21,22,22,22,33,32,33,33,33,33,32,33,44,44,44,45,44,44,44,44},
  {282,12,12,12,12,11,12,12,22,22,22,22,21,22,22,22,33,32,33,33,33,33,32,33,44,44,44,45,44,44,44,44},
  {287,12,13,12,12,12,12,13,22,21,22,22,22,22,21,22,33,33,33,34,33,33,33,33,45,44,44,44,44,45,44,44},
  {287,12,13,12,12,12,12,13,22,21,22,22,22,22,21,22,33,33,33,34,33,33,33,33,45,44,44,44,44,45,44,44},
  {291,12,12,12,13,12,12,12,22,22,22,23,22,22,22,22,34,33,33,33,33,34,33,33,44,44,45,44,45,44,44,45},
  {291,12,12,12,13,12,12,12,22,22,22,23,22,22,22,22,34,33,33,33,33,34,33,33,44,44,45,44,45,44,44,45},
  {295,13,12,13,12,12,13,12,23,22,22,22,22,23,22,22,33,33,34,33,34,33,33,34,44,45,44,44,45,44,45,44},
  {295,13,12,13,12,12,13,12,23,22,22,22,22,23,22,22,33,33,34,33,34,33,33,34,44,45,44,44,45,44,45,44},
  {300,12,12,13,12,13,12,12,22,22,23,22,23,22,22,23,33,34,33,33,34,33,34,33,45,45,44,45,44,45,45,44},
  {300,12,12,13,12,13,12,12,22,22,23,22,23,22,22,23,33,34,33,33,34,33,34,33,45,45,44,45,44,45,45,44},
  {304,12,13,12,13,13,12,13,22,23,22,22,23,22,23,22,34,34,33,34,33,34,34,33,45,44,45,45,44,45,44,45},
  {304,12,13,12,13,13,12,13,22,23,22,22,23,22,23,22,34,34,33,34,33,34,34,33,45,44,45,45,44,45,44,45},
  {307,13,13,12,13,12,13,13,23,23,22,23,22,23,23,22,34,33,34,34,33,34,33,34,45,45,45,44,45,45,45,45},
  {307,13,13,12,13,12,13,13,23,23,22,23,22,23,23,22,34,33,34,34,33,34,33,34,45,45,45,44,45,45,45,45},
  {312,13,12,13,13,13,13,12,23,22,23,23,22,23,22,23,34,34,34,33,34,34,34,34,44,45,45,45,45,44,45,45},
  {312,13,12,13,13,13,13,12,23,22,23,23,22,23,22,23,34,34,34,33,34,34,34,34,44,45,45,45,45,44,45,45},
  {316,13,13,13,12,13,13,13,23,23,23,22,23,23,23,23,33,34,34,34,34,33,34,34,45,45,46,45,45,45,45,46},
  {316,13,13,13,12,13,13,13,23,23,23,22,23,23,23,23,33,34,34,34,34,33,34,34,45,45,46,45,45,45,45,46},
  {320,14,13,13,13,13,14,13,22,23,23,23,23,22,23,23,34,34,35,34,34,34,34,35,45,45,45,45,46,45,45,45},
  {320,14,13,13,13,13,14,13,22,23,23,23,23,22,23,23,34,34,35,34,34,34,34,35,45,45,45,45,46,45,45,45},
  {324,13,13,14,13,13,13,13,23,23,24,23,23,23,23,24,34,34,34,34,35,34,34,34,45,46,45,46,45,45,46,45},
  {324,13,13,14,13,13,13,13,23,23,24,23,23,23,23,24,34,34,34,34,35,34,34,34,45,46,45,46,45,45,46,45},
  {329,13,14,13,13,14,13,14,23,23,23,23,24,23,23,23,34,35,34,35,34,34,35,34,46,45,45,46,45,46,45,45},
  {329,13,14,13,13,14,13,14,23,23,23,23,24,23,23,23,34,35,34,35,34,34,35,34,46,45,45,46,45,46,45,45},
  {332,13,14,13,14,13,13,14,23,24,23,24,23,23,24,23,35,34,34,35,34,35,34,34,46,45,46,45,46,46,45,46},
  {332,13,14,13,14,13,13,14,23,24,23,24,23,23,24,23,35,34,34,35,34,35,34,34,46,45,46,45,46,46,45,46},
  {336,14,13,14,14,13,14,13,24,23,23,24,23,24,23,23,35,34,35,34,35,35,34,35,45,46,46,45,46,45,46,46},
  {336,14,13,14,14,13,14,13,24,23,23,24,23,24,23,23,35,34,35,34,35,35,34,35,45,46,46,45,46,45,46,46},
  {341,14,13,14,13,14,14,13,24,23,24,23,24,24,23,24,34,35,35,34,35,34,35,35,46,46,45,46,46,46,46,45},
  {341,14,13,14,13,14,14,13,24,23,24,23,24,24,23,24,34,35,35,34,35,34,35,35,46,46,45,46,46,46,46,45},
  {345,13,14,14,14,14,13,14,23,24,24,23,24,23,24,24,35,35,34,35,35,35,35,34,46,46,46,46,45,46,46,46},
  {345,13,14,14,14,14,13,14,23,24,24,23,24,23,24,24,35,35,34,35,35,35,35,34,46,46,46,46,45,46,46,46},
  {349,14,14,13,14,14,14,14,24,24,23,24,24,24,24,23,35,35,35,35,34,35,35,35,46,47,46,46,46,46,47,46},
  {349,14,14,13,14,14,14,14,24,24,23,24,24,24,24,23,35,35,35,35,34,35,35,35,46,47,46,46,46,46,47,46},
  {354,14,14,14,14,15,14,14,24,24,24,24,23,24,24,24,35,36,35,35,35,35,36,35,46,46,46,47,46,46,46,46},
  {354,14,14,14,14,15,14,14,24,24,24,24,23,24,24,24,35,36,35,35,35,35,36,35,46,46,46,47,46,46,46,46},
  {357,14,15,14,14,14,14,15,24,25,24,24,24,24,25,24,35,35,35,36,35,35,35,35,47,46,47,46,46,47,46,47},
  {357,14,15,14,14,14,14,15,24,25,24,24,24,24,25,24,35,35,35,36,35,35,35,35,47,46,47,46,46,47,46,47},
  {361,15,14,14,15,14,15,14,24,24,24,25,24,24,24,24,36,35,36,35,35,36,35,36,46,46,47,46,47,46,46,47},
  {361,15,14,14,15,14,15,14,24,24,24,25,24,24,24,24,36,35,36,35,35,36,35,36,46,46,47,46,47,46,46,47},
  {365,15,14,15,14,14,15,14,25,24,25,24,24,25,24,25,35,35,36,35,36,35,35,36,46,47,46,47,47,46,47,46},
  {365,15,14,15,14,14,15,14,25,24,25,24,24,25,24,25,35,35,36,35,36,35,35,36,46,47,46,47,47,46,47,46},
  {370,14,15,15,14,15,14,15,24,24,25,24,25,24,24,25,35,36,35,36,36,35,36,35,47,47,46,47,46,47,47,46},
  {370,14,15,15,14,15,14,15,24,24,25,24,25,24,24,25,35,36,35,36,36,35,36,35,47,47,46,47,46,47,47,46},
  {374,14,15,14,15,15,14,15,24,25,24,25,25,24,25,24,36,36,35,36,35,36,36,35,47,46,47,47,47,47,46,47},
  {374,14,15,14,15,15,14,15,24,25,24,25,25,24,25,24,36,36,35,36,35,36,36,35,47,46,47,47,47,47,46,47},
  {377,15,15,15,15,14,15,15,25,25,24,25,24,25,25,24,36,35,36,36,36,36,35,36,47,47,47,46,47,47,47,47},
  {377,15,15,15,15,14,15,15,25,25,24,25,24,25,25,24,36,35,36,36,36,36,35,36,47,47,47,46,47,47,47,47},
  {382,15,14,15,15,15,15,14,25,24,25,25,25,25,24,25,36,36,36,35,36,36,36,36,48,47,47,47,47,48,47,47},
  {382,15,14,15,15,15,15,14,25,24,25,25,25,25,24,25,36,36,36,35,36,36,36,36,48,47,47,47,47,48,47,47},
  {386,15,15,15,16,15,15,15,25,25,25,24,25,25,25,25,37,36,36,36,36,37,36,36,47,47,48,47,47,47,47,48},
  {386,15,15,15,16,15,15,15,25,25,25,24,25,25,25,25,37,36,36,36,36,37,36,36,47,47,48,47,47,47,47,48},
  {390,16,15,15,15,15,16,15,26,25,25,25,25,26,25,25,36,36,37,36,36,36,36,37,47,48,47,47,48,47,48,47},
  {390,16,15,15,15,15,16,15,26,25,25,25,25,26,25,25,36,36,37,36,36,36,36,37,47,48,47,47,48,47,48,47},
  {395,15,15,16,15,16,15,15,25,25,26,25,25,25,25,26,36,37,36,36,37,36,37,36,47,48,47,48,47,47,48,47},
  {395,15,15,16,15,16,15,15,25,25,26,25,25,25,25,26,36,37,36,36,37,36,37,36,47,48,47,48,47,47,48,47},
  {399,15,16,15,15,16,15,16,25,26,25,25,26,25,26,25,36,37,36,37,36,36,37,36,48,47,48,48,47,48,47,48},
  {399,15,16,15,15,16,15,16,25,26,25,25,26,25,26,25,36,37,36,37,36,36,37,36,48,47,48,48,47,48,47,48},
  {402,16,16,15,16,15,16,16,25,26,25,26,25,25,26,25,37,36,37,37,36,37,36,37,48,47,48,47,48,48,47,48},
  {402,16,16,15,16,15,16,16,25,26,25,26,25,25,26,25,37,36,37,37,36,37,36,37,48,47,48,47,48,48,47,48},
  {406,16,15,16,16,15,16,15,26,25,26,26,25,26,25,26,37,36,37,36,37,37,36,37,47,48,48,48,48,47,48,48},
  {406,16,15,16,16,15,16,15,26,25,26,26,25,26,25,26,37,36,37,36,37,37,36,37,47,48,48,48,48,47,48,48},
  {411,16,16,16,15,16,16,16,26,25,26,25,26,26,25,26,36,37,37,37,37,36,37,37,48,48,47,48,48,48,48,47},
  {411,16,16,16,15,16,16,16,26,25,26,25,26,26,25,26,36,37,37,37,37,36,37,37,48,48,47,48,48,48,48,47},
  {415,15,16,16,16,16,15,16,25,26,26,26,26,25,26,26,37,37,36,37,37,37,37,36,48,48,48,48,49,48,48,48},
  {415,15,16,16,16,16,15,16,25,26,26,26,26,25,26,26,37,37,36,37,37,37,37,36,48,48,48,48,49,48,48,48},
  {419,16,16,17,16,16,16,16,26,26,25,26,26,26,26,25,37,37,37,37,38,37,37,37,48,49,48,48,48,48,49,48},
  {419,16,16,17,16,16,16,16,26,26,25,26,26,26,26,25,37,37,37,37,38,37,37,37,48,49,48,48,48,48,49,48},
  {424,16,16,16,16,17,16,16,26,26,26,26,27,26,26,26,37,38,37,37,37,37,38,37,49,48,48,49,48,49,48,48},
  {424,16,16,16,16,17,16,16,26,26,26,26,27,26,26,26,37,38,37,37,37,37,38,37,49,48,48,49,48,49,48,48},
  {427,16,17,16,17,16,16,17,26,27,26,26,26,26,27,26,38,37,37,38,37,38,37,37,49,48,49,48,48,49,48,49},
  {427,16,17,16,17,16,16,17,26,27,26,26,26,26,27,26,38,37,37,38,37,38,37,37,49,48,49,48,48,49,48,49},
  {431,17,16,16,17,16,17,16,27,26,26,27,26,27,26,26,38,37,38,37,37,38,37,38,48,49,49,48,49,48,49,49},
  {431,17,16,16,17,16,17,16,27,26,26,27,26,27,26,26,38,37,38,37,37,38,37,38,48,49,49,48,49,48,49,49},
  {436,17,16,17,16,17,17,16,27,26,27,26,26,27,26,27,37,38,38,37,38,37,38,38,48,49,48,49,49,48,49,48},
  };

void cMap0101::MakePrime(BIGNUM *n, unsigned char *residues)
{
  bool isPrime;
  cycles+=1290;
  do {
    cycles+=1465;
    BN_add_word(n,2);
    isPrime=true;
    for(int i=0; i<53; i++) {
      residues[i]+=2;
      unsigned char num=residues[i];
      unsigned char denom=primes[i];
      if(num>denom) {
        unsigned char r=0;
        while(denom>=r) { cycles+=1; r=(r<<1)|((num&0x80)>>7); num<<=1; }
        }
      residues[i]%=primes[i];
      if(residues[i]==0) { cycles+=13; isPrime=false; }
      }
    } while(!isPrime);
}

void cMap0101::DoMap(int f, unsigned char *data, int l)
{
  PRINTF(L_SYS_MAP,"0101: calling function %02X",f);
  l=GetOpSize(l);
  cycles=0;
  switch(f) {
    case 0x22:
      {
      int shift=((-l)&0xFF00)>>8, exp=(-l)&0x1F;
      BN_zero(B);
      BN_set_bit(B,wordsize<<6);
      BN_sub(B,B,D);
      BN_set_word(I,exp*2);
      BN_mod_exp(B,B,I,D,ctx);
      BN_mod_lshift(B,B,shift,D,ctx);
      if(exp) cycles=559 + 2112*exp + 16*((shift+1)>>1) + 4*coef22[shift][exp] - 6;
      else    cycles=931 +                                4*coef22[shift][0]   - 6;
      break;
      }
    case 0x29:
      {
      BN_add(B,B,C);
      bool b=BN_is_bit_set(B,wordsize<<6);
      if(data) data[0]=b;
      if(b) BN_mask_bits(B,wordsize<<6);
      cycles=501+(8*wordsize+3)/5*5-6;
      }
      break; 
    case 0x32:
      l=min(34,l);
      A.GetLE(data,l<<3);
      BN_div(C,B,A,D,ctx);
      BN_rshift(A,C,17*64);
      BN_mask_bits(C,17*64);
      BN_zero(J);
      break;
    case 0x3b:
      MonInit(wordsize*60+4*l);
      I.GetLE(data,l<<3);
      MonMul(B,I,B,l);
      break;
    case 0x3e:
      {
      I.GetLE(data,l<<3);
      BN_mod_exp(B,A,I,D,ctx);
      BN_one(A);
      int end=BN_num_bits(I);
      int msb=data[(end-1)/8];
      cycles=3848 + ((end-1)/8)*650 - 11;
      for(int i=8; --i>=1;) if(msb&(1<<i)) { cycles+=(i*75)-15; break; }
      for(int i=end; --i>=0;) if(BN_is_bit_set(I,i)) cycles+=88;
      break;
      }
    case 0x4d:
      if(-0x018000==l)
        BN_mask_bits(B,64);
      else {
        BN_set_bit(B,(wordsize<<6)-1);
        if(-0x028000==l) BN_set_bit(B,(wordsize<<6)-2);
        }
      BN_set_bit(B,0);
      for(int i=0; i<53; i++) data[i]=BN_mod_word(B,primes[i]);
      break;
    case 0x4e:
      MakePrime(B,data);
      break;
    case 0x57:
#ifdef HAS_AUXSRV
      {
      int c=aux.Map(0x57,data,0x60,0x40);
      if(c>0) { cycles=c-6; break; }
      }
#endif
      {
      cBN a, b, x, y, scalar;
      if(l<2 || l>4) l=4;
      TMPWS_START(l);
      l<<=3;
      D.GetLE(data+0*l,l);
      x.GetLE(data+1*l,l);
      y.GetLE(data+2*l,l);
      b.GetLE(data+3*l,l);
      a.GetLE(data+4*l,l);
      scalar.GetLE(data+5*l,l);
      bool doz=false;
      int scalarbits=BN_num_bits(scalar);
      if(scalarbits>=2) {
        if(BN_is_zero(x) && (BN_is_zero(y) || (BN_is_zero(b) && BN_num_bits(y)==1))) {
          BN_zero(Px);
          BN_copy(Py,y);
          BN_zero(Qz);
          }
        else {
          CurveInit(a);
          ToProjective(0,x,y);
          BN_copy(Qx,Px);
          BN_copy(Qy,Py);
          for(int i=scalarbits-2; i>=0; i--) {
            DoubleP(0);
            if(BN_is_bit_set(scalar,i)) {
              BN_copy(A,Pz);
              if(BN_is_zero(Pz) || BN_is_zero(D)) {
                BN_copy(Px,Qx);
                BN_copy(Py,Qy);
                BN_copy(Pz,Qz);
                AddP(1);
                }
              else {
                doz=true;
                if(wordsize==4) {
                  BN_rshift(Py,Py,32);
                  BN_lshift(Py,Py,32);
                  BN_rshift(b,Qz,224);
                  BN_add(Py,Py,b);
                  }
                BN_mask_bits(Px,32);
                BN_lshift(b,Qz,32);
                BN_add(Px,Px,b);
                BN_mask_bits(Px,128);
                AddP(0);
                }
              }
            }
          ToAffine();
          }
        }
      else {
        BN_copy(Px,x);
        BN_copy(Py,y);
        BN_zero(Qz);
        }
      memset(data,0,0x40);
      Px.PutLE(&data[0x00],l);
      if(l<0x20 && doz) {
        unsigned char tmp[0x20];
        Qz.PutLE(tmp,l);
        memcpy(&data[l],&tmp[l-4],4);
        }
      Py.PutLE(&data[0x20],l);
      TMPWS_END();
      BN_zero(A);
      BN_zero(B);
      BN_zero(C);
      break;
      }
    default:
      if(!cMapCore::DoMap(f,data,l))
        PRINTF(L_SYS_MAP,"0101: unsupported call %02x",f);
      break;
    }
}

// -- cN2Prov0101 --------------------------------------------------------------

class cN2Prov0101 : public cN2Prov, public cN2Emu, private cMap0101 {
private:
  int desSize;
  DES_key_schedule desks1, desks2;
  unsigned char desblock[8];
  IdeaKS ks;
  cMapMemHW *hwMapper;
  //
  void AddRomCallbacks(void);
  bool RomCallbacks(void);
  bool ProcessMap(int f);
  bool ProcessDESMap(int f);
protected:
  int mecmAddr[2];
  int mecmKeyId;
  //
  virtual bool Algo(int algo, const unsigned char *hd, unsigned char *hw);
  virtual void DynamicHD(unsigned char *hd, const unsigned char *ed);
  virtual bool RomInit(void);
  virtual void Stepper(void);
  virtual void TimerHandler(unsigned int num);
public:
  cN2Prov0101(int Id, int Flags);
  virtual bool PostProcAU(int id, unsigned char *data);
  virtual int ProcessBx(unsigned char *data, int len, int pos);
  virtual int ProcessEx(unsigned char *data, int len, int pos);
  virtual int RunEmu(unsigned char *data, int len, unsigned short load, unsigned short run, unsigned short stop, unsigned short fetch, int fetch_len);
  };

static cN2ProvLinkReg<cN2Prov0101,0x0101,(N2FLAG_MECM|N2FLAG_POSTAU|N2FLAG_Bx|N2FLAG_Ex)> staticPL0101;

cN2Prov0101::cN2Prov0101(int Id, int Flags)
:cN2Prov(Id,Flags)
{
  mecmAddr[0]=0x91d7; mecmAddr[1]=0x92d7; mecmKeyId=0x106;
  seedSize=10;
  desSize=16; hwMapper=0;
}

void cN2Prov0101::DynamicHD(unsigned char *hd, const unsigned char *ed)
{
  hd[5]=ed[5];
  hd[6]=(ed[7]&0xEF) | ((ed[6]&0x40)>>2);
  hd[7]=ed[8];
  hd[8]=(ed[9]&0x7F) | ((ed[6]&0x20)<<2);
  hd[9]=ed[6]&0x80;
}

bool cN2Prov0101::Algo(int algo, const unsigned char *hd, unsigned char *hw)
{
  if(algo!=0x40 && algo!=0x60) {
    PRINTF(L_SYS_ECM,"%04X: unknown MECM algo %02x",id,algo);
    return false;
    }
  if(!Init(id,102)) {
    PRINTF(L_SYS_ECM,"%04X: failed to initialize ROM",id);
    return false;
    }

  unsigned char keyNr=(algo>>5)&0x01;
  unsigned char mecmCode[256];
  GetMem(mecmAddr[keyNr],mecmCode,256,0x80);
  cPlainKey *pk;
  unsigned char ideaKey[16];
  if(!(pk=keys.FindKey('N',mecmKeyId,keyNr,sizeof(ideaKey)))) {
    PRINTF(L_SYS_KEY,"missing %04x %02x MECM key",mecmKeyId,keyNr);
    return false;
    }
  pk->Get(ideaKey);
  idea.SetEncKey(ideaKey,&ks);
  for(int i=0x100-8; i>=8; i-=8) {
    idea.Encrypt(mecmCode+i,8,mecmCode+i,&ks,0);
    xxor(mecmCode+i,8,mecmCode+i,mecmCode+i-8);
    }
  idea.Encrypt(mecmCode,8,mecmCode,&ks,0);
  HEXDUMP(L_SYS_RAWECM,mecmCode,sizeof(mecmCode),"decrypted MECM code");
  // check signature
  unsigned char data[256];
  memcpy(data,mecmCode,sizeof(data));
  RotateBytes(data,sizeof(data));
  SHA1(data,sizeof(data)-8,data);
  RotateBytes(data,20);
  if(memcmp(data,mecmCode,8)) {
     PRINTF(L_SYS_ECM,"%04X: MECM %02x decrypt signature failed",id,keyNr);
     return false;
    }

  memcpy(hw,hd,seedSize);
  ExpandInput(hw);

  SetSp(0x0FFF,0x0EF8);
  ClearBreakpoints();
  SetMem(0x0100,mecmCode+8,0x100-8);
  SetMem(0xa00,hd,seedSize);
  SetMem(0x0ba2,hw,0x80);
  AddBreakpoint(0x0000);
  AddRomCallbacks();
  SetPc(0x0100);
  while(!Run(100000)) {
    if(GetPc()==0x0000) {
      GetMem(0x0ba2,hw,0x80);
      return true;
      }
    else if(!RomCallbacks()) return false;
    }

/*
  if(algo==0x40) {
    const int hwCount=2;
    memcpy(hw,hd,hwCount+1);
    ExpandInput(hw);
    hw[0x18]|=1; hw[0x40]|=1;
    DoMap(SETSIZE,0,2);
    DoMap(IMPORT_A,hw,3);
    DoMap(IMPORT_D,hw+24);
    DoMap(0x3b);
    DoMap(EXPORT_C,hw);
    DoMap(IMPORT_A,hw+40,3);
    DoMap(IMPORT_D,hw+64);
    DoMap(0x3b);
    DoMap(EXPORT_C,hw+40);
    DoMap(0x43);
    DoMap(0x44,hw);
    DoMap(0x44,hw+64);
    memcpy(&hw[0],&hw[64+4],hwCount);
    memset(&hw[hwCount],0,128-hwCount);
    return true;
    }
  else if(algo==0x60) { // map 4D/4E/57
    hw[127]|=0x80; hw[0]|=0x01;
    DoMap(SETSIZE,0,16);
    DoMap(IMPORT_A,hw);
    DoMap(0x4d);
    DoMap(0x4e);
    DoMap(EXPORT_A,hw);
    RotateBytes(hw,16);
    DoMap(IMPORT_A,hw);
    DoMap(0x4e);
    DoMap(EXPORT_A,hw);
    DoMap(0x57,hw);
    return true;
    }

  PRINTF(L_SYS_ECM,"%04X: unknown MECM algo %02x",id,algo);
*/
  return false;
}

bool cN2Prov0101::PostProcAU(int id, unsigned char *data)
{
  if(data[1]&0x5f) return false;
  return true;
}

bool cN2Prov0101::RomInit(void)
{
  if(!AddMapper(hwMapper=new cMapMemHW(),HW_OFFSET,HW_REGS,0x00)) return false;
  SetPc(0x4000);
  SetSp(0x0FFF,0x0FE0);
  ClearBreakpoints();
  AddBreakpoint(0x537d);
  AddBreakpoint(0x8992);
  AddBreakpoint(0xA822);
  while(!Run(5000)) {
    switch(GetPc()) {
      case 0x537d:
        PRINTF(L_SYS_EMU,"%04x: ROM init successfull",id);
        return true;
      default:
        PRINTF(L_SYS_EMU,"%04x: ROM init failed: unexpected breakpoint",id);
        break;
      }
    }
  return false;
}

bool cN2Prov0101::ProcessMap(int f)
{
  unsigned short addr;
  unsigned char tmp[512];
  int l=GetOpSize(Get(0x48));
  int dl=l<<3;

  switch(f) {
    case SETSIZE:
      DoMap(f,0,Get(0x48));
      AddCycles(MapCycles());
      break;
    case IMPORT_J:
    case IMPORT_A:
    case IMPORT_B:
    case IMPORT_C:
    case IMPORT_D:
    case IMPORT_LAST:
      addr=HILO(0x44);
      GetMem(addr,tmp,dl,0); DoMap(f,tmp,l);
      AddCycles(MapCycles());
      break;
    case EXPORT_J:
    case EXPORT_A:
    case EXPORT_B:
    case EXPORT_C:
    case EXPORT_D:
    case EXPORT_LAST:
      addr=HILO(0x44);
      DoMap(f,tmp,l); SetMem(addr,tmp,dl,0);
      AddCycles(MapCycles());
      break;
    case SWAP_A:
    case SWAP_B:
    case SWAP_C:
    case SWAP_D:
      addr=HILO(0x44);
      GetMem(addr,tmp,dl,0); DoMap(f,tmp,l); SetMem(addr,tmp,dl,0);
      AddCycles(MapCycles());
      break;
    case CLEAR_A:
    case CLEAR_B:
    case CLEAR_C:
    case CLEAR_D:
    case COPY_A_B:
    case COPY_B_A:
    case COPY_A_C:
    case COPY_C_A:
    case COPY_C_D:
    case COPY_D_C:
      DoMap(f);
      AddCycles(MapCycles());
      break;
    case 0x22:
      DoMap(f,tmp,-((Get(0x48)<<16)|(Get(0x49)<<8)|Get(0x4a)));
      AddCycles(MapCycles());
      break;
    case 0x29:
      GetMem(HILO(0x44),tmp,dl,0);
      DoMap(f,tmp);
      Set(0x4b,tmp[0]);
      AddCycles(MapCycles());
      break;
    case 0x3e:
      if(l>wordsize) { l=wordsize; dl=l<<3; }
      // fall through
    case 0x32:
    case 0x3b:
      if(l>34) { l=34; dl=34<<3; }
      GetMem(HILO(0x44),tmp,dl,0);
      DoMap(f,tmp,l);
      AddCycles(MapCycles());
      break;
    case 0x43:
      DoMap(f);
      break;
    case 0x44:
    case 0x45:
      GetMem(0x400,tmp,64,0);
      DoMap(f,tmp,l);
      SetMem(0x440,tmp,20,0);
      break; 
    case 0x4d:
      DoMap(f,tmp,-((Get(0x48)<<16)|(Get(0x49)<<8)|Get(0x4a)));
      SetMem(0x400,tmp,53,0);
      break;
    case 0x4e:
      GetMem(0x400,tmp,53,0);
      DoMap(f,tmp);
      SetMem(0x400,tmp,53,0);
      AddCycles(MapCycles());
      break;
    case 0x57:
      addr=HILO(0x46);
      l=wordsize; if(l<2 || l>4) l=4;
      dl=l<<3;
      for(int i=0; i<6; i++) GetMem(HILO(addr+i*2),tmp+i*dl,dl,0);
      DoMap(f,tmp);
      SetMem(0x400,tmp,0x40,0);
      memset(tmp,0,11*32);
      SetMem(0x440,tmp,11*32,0);
      AddCycles(MapCycles());
      break;
    default:
      PRINTF(L_SYS_EMU,"%04x: map call %02x not emulated",id,f);
      return false;
    }
  c6805::a=0; c6805::x=0; c6805::y=0; 
  return true;
}

bool cN2Prov0101::ProcessDESMap(int f)
{
  unsigned char data[16];
  switch(f) {
    case 0x05:  // 3DES encrypt
      DES_ecb2_encrypt(DES_CAST(desblock),DES_CAST(desblock),&desks1,&desks2,DES_ENCRYPT);
      break;
    case 0x06:  // 3DES decrypt
      DES_ecb2_encrypt(DES_CAST(desblock),DES_CAST(desblock),&desks1,&desks2,DES_DECRYPT);
      break;
    case 0x0b:  // load DES data block from memory
      GetMem(HILO(0x25),desblock,8,Get(0x24));
      break;
    case 0x0c:  // store DES data block to memory
      SetMem(HILO(0x2b),desblock,8,Get(0x2a));
      break;
    case 0x0e:  // get DES key1 and key2
      GetMem(HILO(0x25),data,8,Get(0x24));
      DES_key_sched((DES_cblock *)data,&desks1);
      GetMem(HILO(0x28),data,8,Get(0x27));
      DES_key_sched((DES_cblock *)data,&desks2);
      break;
    case 0x0f:  // set DES size
      desSize=Get(0x2d);
      if(desSize!=0x10 && desSize!=0x18) {
        PRINTF(L_SYS_EMU,"%04x: invalid DES key size %02x",id,desSize);
        return false;
        }
      break;
    default:
      PRINTF(L_SYS_EMU,"%04x: DES map call %02x not emulated",id,f);
      return false;
  }
  return true;
}

bool cN2Prov0101::RomCallbacks(void)
{
  unsigned int ea=GetPc();
  if(ea&0x8000) ea|=(cr<<16);
  switch(ea) {
    case 0x3840: //MAP Handler
    case 0x00A822:
      if(!ProcessMap(a)) return false;
      break;
    case 0x3844: //DES Handler
      if(!ProcessDESMap(a)) return false;
      break;
    case 0x5F23: //Erase_RAM_and_Hang
    case 0x5F27: //Erase_RAM_and_Hang_Lp
    case 0x5F5E: //BrainDead
      PRINTF(L_SYS_EMU,"%04x: emu hung at %04x",id,ea);
      return false;
    case 0x70A6: //Write_Row_EEP_RC2_Len_A_To_RC1
      {
      unsigned short rc1=HILO(0x47);
      unsigned short rc2=HILO(0x4a);
      for(int i=0; i<a; i++) Set(0x80,rc1++,Get(dr,rc2++));
      Set(0x4a,rc2>>8);
      Set(0x4b,rc2&0xff);
      break;
      }
    case 0x7BFE: //Write_Row_EEPROM_A_from_X_to_RC1
      {
      unsigned short rc1=HILO(0x47);
      unsigned short rc2=c6805::x;
      for(int i=0; i<a; i++) Set(0x80,rc1++,Get(rc2++));
      break;
      }
    case 0x7CFF: //UPDATE_USW_03DD_03DE
      Set(0x30E8,Get(0x03DD));
      Set(0x30E9,Get(0x03DE));
      break;
    case 0x00A23C: //IDEA_Generate_Expanded_Key
      {
      unsigned char key[16];
      GetMem(0x0a20,key,16);
      idea.SetEncKey(key,&ks);
      break;
      }
    case 0x00A2E9: //IDEA_Cypher
      {
      unsigned char data[8];
      GetMem(0x070,data,8);
      idea.Encrypt(data,8,data,&ks,0);
      SetMem(0x070,data,8);
      break;
      }
    default:
      PRINTF(L_SYS_EMU,"%04X: unknown ROM breakpoint %04x",id,ea);
      return false;
    }
  if(ea>=0x8000) PopCr();
  PopPc();
  return true;
}

void cN2Prov0101::AddRomCallbacks(void)
{
  AddBreakpoint(0xA822); // map handler
  AddBreakpoint(0x3840);
  AddBreakpoint(0x3844);
  AddBreakpoint(0xA23C); //IDEA 
  AddBreakpoint(0xA2E9);
  AddBreakpoint(0x70A6);
  AddBreakpoint(0x7BFE);
  AddBreakpoint(0x7CFF);
  AddBreakpoint(0x5F23);
  AddBreakpoint(0x5F27);
  AddBreakpoint(0x5F5E);
}

int cN2Prov0101::ProcessBx(unsigned char *data, int len, int pos)
{
  if(Init(id,102)) {
    SetMem(0x92,data+pos-1,len-pos+1);
    SetPc(0x93);
    SetSp(0x0FFF,0x0EF8);
    ClearBreakpoints();
    AddBreakpoint(0x9569);
    AddBreakpoint(0x0000);
    AddRomCallbacks();
    while(!Run(1000)) {
      if(GetPc()==0x9569) {
        GetMem(0x80,data,len);
        return a;
        }
      else if(GetPc()==0x0000) break;
      else if(!RomCallbacks()) break;
      }
    }
  return -1;
}

int cN2Prov0101::ProcessEx(unsigned char *data, int len, int pos)
{
  if(Init(id,102)) {
    SetMem(0x80,data,len);
    SetPc(0x9591);
    SetSp(0x0FFF,0x0EF8);
    Push(0x99); //push the bug-table return onto the stack
    Push(0x95);
    Push(0x00);
    Set(0x00,0x62,0x26);
    Set(0x00,0x63,0x02);
    Set(0x00,0x03d3,len-0x12);
    ClearBreakpoints();
    AddBreakpoint(0x9569);
    AddBreakpoint(0x9599);
    AddRomCallbacks();
    while(!Run(10000)) {
      if(GetPc()==0x9569) {
        GetMem(0x80,data,len);
        return max((int)a,6);
        }
      else if(GetPc()==0x9599) break;
      else if(!RomCallbacks()) break;
      }
    }
  return -1;
}

int cN2Prov0101::RunEmu(unsigned char *data, int len, unsigned short load, unsigned short run, unsigned short stop, unsigned short fetch, int fetch_len)
{
  if(Init(id,102)) {
    SetSp(0x0FFF,0x0EF8);
    SetMem(load,data,len);
    SetPc(run);
    ClearBreakpoints();
    AddBreakpoint(stop);
    if(stop!=0x0000) AddBreakpoint(0x0000);
    AddRomCallbacks();
    while(!Run(100000)) {
      if(GetPc()==0x0000 || GetPc()==stop) {
        GetMem(fetch,data,fetch_len);
        return 1;
        }
      else if(!RomCallbacks()) break;
      }
    }
  return -1;
}

void cN2Prov0101::TimerHandler(unsigned int num)
{
  if(hwMapper) hwMapper->AddCycles(num);
}

void cN2Prov0101::Stepper(void)
{
  unsigned short pc=GetPc();
  if(pc>=0x93 && pc<=0xE0) { 	// pc in EMM data
    unsigned char op=Get(pc);
    if((op&0xF0)==0x00) { 	// current opcode BRCLR/BRSET
      int fake=0; 		// 1=branch -1=non-branch
      if(Get(pc+3)==0x81) 	// next opcode == RTS
        fake=1;			// fake branch
      else {
        unsigned char off=Get(pc+2);
        unsigned short target=pc+3+off;
        if(off&0x80) target-=0x100;
        if(Get(target)==0x81) 	// branch target == RTS
          fake=-1;		// fake non-branch
        }
      if(fake) {
        unsigned short ea=Get(pc+1);
        unsigned char val=Get(dr,ea);
        int bit=1<<((op&0xF)>>1);
        // set/clr bit according to fake-mode and opcode
        if((fake>0 && (op&0x01)) || (fake<0 && !(op&0x01)))  {
          if(val&bit) loglb->Printf("*");
          val&=~bit;
          }
        else {
          if(!(val&bit)) loglb->Printf("*");
          val|=bit;
          }
        Set(dr,ea,val);
        }
      }
    }
}

// -- cN2Prov0901 --------------------------------------------------------------

class cN2Prov0901 : public cN2Prov0101 {
public:
  cN2Prov0901(int Id, int Flags);
  };

static cN2ProvLinkReg<cN2Prov0901,0x0901,(N2FLAG_MECM|N2FLAG_POSTAU|N2FLAG_Bx|N2FLAG_Ex)> staticPL0901;

cN2Prov0901::cN2Prov0901(int Id, int Flags)
:cN2Prov0101(Id,Flags)
{
  mecmAddr[0]=0x91f5; mecmAddr[1]=0x92f5; mecmKeyId=0x907;
}
