/*
 * freeboxtv.c: A plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id: freeboxtv.c 10 2007-09-11 16:14:11Z odj $
 */

// General includes
#include <time.h>

#ifdef FREEBOXTV_EPG
#include <libxml++/libxml++.h>
#endif

// VDR includes
#include <vdr/plugin.h>
#include <vdr/device.h>
#include <vdr/sources.h>
#include <vdr/ringbuffer.h>

// LiveMedia includes
#include "liveMedia.hh"
#include "BasicUsageEnvironment.hh"
#include "GroupsockHelper.hh"
#include "RTSPClient.hh"

#include "freeboxtv.h"

static const char *VERSION        = "0.0.3";
static const char *DESCRIPTION    = "Watch FreeboxTv with VDR";
static const char *plugName       = "FreeboxTV Plugin";

using namespace FreeboxTv;
using FreeboxTv::cFreeboxClient;
cFreeboxClient *freeboxClient;
using FreeboxTv::cFreeboxEpg;
cFreeboxEpg *FreeboxEpg;

extern char *url;

extern UsageEnvironment *env;
unsigned short ClientPortNum;   // Port client  utiliser : 0 = random value
extern cRingBufferLinear *TSDataBuffer;

extern cMutex fbmutex;

// --- cFreeboxChannel --------------------------------------------------------
class cFreeboxChannel : public cListObject {
private:
  char *fbChannelEpgId;
  int fbChannelId;
  char *fbChannelUrl;
public:
  virtual ~cFreeboxChannel();
  tChannelID channelID;
  char* GetFbChannelEpgId() { return &fbChannelEpgId [0]; }
  int GetFbChannelId() { return fbChannelId; }
  char* GetFbUrl() { return &fbChannelUrl [0]; }
  bool Parse(const char *s);
  };

bool cFreeboxChannel::Parse(const char *s)
{
  char *fbtid = NULL;
  char *idx = NULL;
  char *fbChId = new char [30];

  if (1 == sscanf(s, "%a[^:]", &fbtid)) {
    fbChannelId = atoi (fbtid);

    // Build channel ID (form C-1-XXXX-1-0, where XXXX = NID)
    sprintf (fbChId, "C-1-%s-1-0", fbtid);
    channelID = tChannelID::FromString (fbChId);

    // Get EPG (XMLTV) channel id from channel.conf.freebox corresponding to channel ID
    idx = index (s, ':') + 1;
    sscanf(idx, "%a[^:]", &fbChannelEpgId);

    // Get URL from channel.conf.freebox corresponding to channel ID
    idx = index (idx + 1, ':') + 1;
    fbChannelUrl = (char *)malloc (strlen(idx) + 1);
    strcpy (fbChannelUrl, idx);
  }

  free (fbtid);
  delete fbChId;
  return channelID.Valid();
}

cFreeboxChannel::~cFreeboxChannel()
{
  free (fbChannelUrl);
  fbChannelUrl = NULL;
  free (fbChannelEpgId);
  fbChannelEpgId = NULL;
}

// --- cFreeboxChannels--------------------------------------------------------
class cFreeboxChannels : public cConfig<cFreeboxChannel> {
public:
  cFreeboxChannel *GetFreeboxChannel(const cChannel *Channel);
  char *GetFreeboxUrl (int fbChannelId);
  int GetFreeboxEpgId (const char *xmltv_channel_name);
  };

cFreeboxChannel *cFreeboxChannels::GetFreeboxChannel(const cChannel *Channel)
{
  tChannelID ChannelID = Channel->GetChannelID();

  for (cFreeboxChannel *fc = First(); fc; fc = Next(fc))
    if (ChannelID == fc->channelID)
      return fc;
  return NULL;
}

char* cFreeboxChannels::GetFreeboxUrl(int fbChannelId)
{
  for (cFreeboxChannel *fbn = First(); fbn; fbn = Next(fbn))
      if (fbChannelId == fbn->GetFbChannelId())
         return fbn->GetFbUrl();

  return NULL;
}

int cFreeboxChannels::GetFreeboxEpgId (const char *xmltv_channel_name)
{
  for (cFreeboxChannel *fbn = First(); fbn; fbn = Next(fbn))
    if (!strcmp (xmltv_channel_name, fbn->GetFbChannelEpgId()))
      return fbn->GetFbChannelId();
  return 0;
}


cFreeboxChannels FreeboxChannels;

// --- cFreeboxDevice --------------------------------------------------------
class cFreeboxDevice : public cDevice {
private:
  int source;
protected:
  virtual bool SetPid(cPidHandle *Handle, int Type, bool On);
  virtual bool OpenDvr(void);
  virtual void CloseDvr(void);
  virtual bool GetTSPacket(uchar *&Data);
public:
  cFreeboxDevice(void);
  virtual ~cFreeboxDevice();
  virtual bool ProvidesSource(int Source) const;
  virtual bool ProvidesTransponder(const cChannel *Channel) const;
  virtual bool ProvidesChannel(const cChannel *Channel, int Priority = -1, bool *NeedsSetChannel = NULL) const;
  virtual bool SetChannelDevice(const cChannel *Channel, bool LiveView);
  virtual bool SwitchChannel(cFreeboxChannel *Channel, bool LiveView);
};

cFreeboxDevice::cFreeboxDevice(void)
{
  // On utilise C (cable) comme source en attendant mieux... 
  source = cSource::FromString("C");
}

cFreeboxDevice::~cFreeboxDevice()
{
}

bool cFreeboxDevice::SetPid(cPidHandle *Handle, int Type, bool On)
{
  //dsyslog("SetPid %d %d", Handle->pid, On);
  return true;
}

// Accs au flux
bool cFreeboxDevice::OpenDvr(void)
{
  freeboxClient = new cFreeboxClient;
  freeboxClient->init_session(env);
  return true;
}

void cFreeboxDevice::CloseDvr(void)
{
  freeboxClient->shutdown();
}

// Lecture du flux
bool cFreeboxDevice::GetTSPacket(uchar *&Data)
{
  fbmutex.Lock();
  int nblus = 0;
  Data = TSDataBuffer->Get (nblus); // Read buffer (1 frame = several packets)
  if (Data) {
    TSDataBuffer->Del (TS_SIZE);    // Delete only 1 packet
  }
  fbmutex.Unlock();

// Comme la lecture n'est pas bloquante, on temporise pour viter un poll trop agressif de VDR
  if (!Data)
    cCondWait::SleepMs(10);

  return true;
}

bool cFreeboxDevice::ProvidesSource(int Source) const
{
  return source == Source;
}

bool cFreeboxDevice::ProvidesTransponder(const cChannel *Channel) const
{
  return false; // can't provide any actual transponder
}

// Peux-tu fournir la chane spcifie en entre ? Avec quelle priorit ?
bool cFreeboxDevice::ProvidesChannel(const cChannel *Channel, int Priority, bool *NeedsDetachReceivers) const
{
  bool result = false;
  bool hasPriority = Priority < 0 || Priority > this->Priority();
  bool needsDetachReceivers = true;

  cFreeboxChannel *FreeboxChannel = FreeboxChannels.GetFreeboxChannel(Channel);

  if (FreeboxChannel) {         // Chane dans la liste ?
    if (Receiving(true)) { // Source active ?
      // Oui, on ne fait rien
      needsDetachReceivers = false;
      result = true;
    }
    // Je peux le faire si je suis le plus prioritaire
    else
      result = hasPriority;
  }


  if (NeedsDetachReceivers)
    // Si je ne fournis pas le service, il faut changer la source
    *NeedsDetachReceivers = needsDetachReceivers;

  return result;
}

bool cFreeboxDevice::SetChannelDevice(const cChannel *Channel, bool LiveView)
{
  if (!Receiving(true)) { // if we are receiving the channel is already set!
     cFreeboxChannel *FreeboxChannel = FreeboxChannels.GetFreeboxChannel(Channel);
     if (FreeboxChannel) {
        // Changer l'URL de lecture du flux
        if (!SwitchChannel(FreeboxChannel, LiveView))
            return false;
     }
  }
  return true;
}

bool cFreeboxDevice::SwitchChannel (cFreeboxChannel *Channel, bool LiveView)
{
  strcpy (url, Channel->GetFbUrl());

  return true;
}

// --- cMenuSetupFreeboxTv -------------------------------------------------------

class cMenuSetupFreeboxTv : public cMenuSetupPage {
private:
  int newClientPortNum;
protected:
  virtual void Store(void);
public:
  cMenuSetupFreeboxTv(void);
  };

cMenuSetupFreeboxTv::cMenuSetupFreeboxTv(void)
{
  newClientPortNum = ClientPortNum;
  Add(new cMenuEditIntItem( tr("Client Port Number (0 = random) "), &newClientPortNum));
}

void cMenuSetupFreeboxTv::Store(void)
{
  SetupStore("ClientPortNum", ClientPortNum = newClientPortNum);
}

// --- cPluginFreeboxTv-------------------------------------------------------
class cPluginFreeboxTv : public cPlugin {
private:
  // Add any member variables or functions you may need here.
public:

  cPluginFreeboxTv(void);
  virtual ~cPluginFreeboxTv();
  virtual const char *Version(void) { return VERSION; }
  virtual const char *Description(void) { return DESCRIPTION; }
  virtual const char *CommandLineHelp(void);
  virtual bool ProcessArgs(int argc, char *argv[]);
  virtual bool Initialize(void);
  virtual void Housekeeping(void);
  virtual cMenuSetupPage *SetupMenu(void);
  virtual bool SetupParse(const char *Name, const char *Value);
  virtual const char **SVDRPHelpPages(void);
  virtual cString SVDRPCommand(const char *Command, const char *Option, int &ReplyCode);

  };

cPluginFreeboxTv::cPluginFreeboxTv(void)
{
  // Initialize any member variables here.
  // DON'T DO ANYTHING ELSE THAT MAY HAVE SIDE EFFECTS, REQUIRE GLOBAL
  // VDR OBJECTS TO EXIST OR PRODUCE ANY OUTPUT!
}

cPluginFreeboxTv::~cPluginFreeboxTv()
{
  // Clean up after yourself!
}

const char *cPluginFreeboxTv::CommandLineHelp(void)
{
  // Return a string that describes all known command line options.
  return NULL;
}

bool cPluginFreeboxTv::ProcessArgs(int argc, char *argv[])
{
  // Implement command line argument processing here if applicable.
  return true;
}

bool cPluginFreeboxTv::Initialize(void)
{
  // Initialize any background activities the plugin shall perform.
  if (FreeboxChannels.Load(AddDirectory(ConfigDirectory(Name()), "channels.conf.freebox"), true, true)) {
    new cFreeboxDevice;
  }
  else {
    esyslog("ERROR: error reading channels.conf.freebox file");
    return false;
  }

#ifdef FREEBOXTV_EPG
  FreeboxEpg = new cFreeboxEpg;
  FreeboxEpg->SetLastGrab(0);
  FreeboxEpg->SetLastEpgUpdate(0);

  // Read and parse XMLTV file
  if (FreeboxEpg->Parse(AddDirectory(ConfigDirectory(Name()), "xmltv.freebox"))) {
    FreeboxEpg->SetLastEpgUpdate();
  }
  else {
    esyslog("ERROR: error reading xmltv.freebox file");
  }
#endif

  // No or bad EPG is not a fatal error
  return true;
}

void cPluginFreeboxTv::Housekeeping(void)
{
  // Perform any cleanup or other regular tasks.

#ifdef FREEBOXTV_EPG
  // Mise  jour des donnes EPG de VDR si cela n'a pas t fait depuis au moins 6 heures
  if ((int)(time(NULL) - FreeboxEpg->GetLastEpgUpdate()) > 21600)
      if (FreeboxEpg->Parse(AddDirectory(ConfigDirectory(Name()), "xmltv.freebox")))
        FreeboxEpg->SetLastEpgUpdate();
#endif
}

cMenuSetupPage *cPluginFreeboxTv::SetupMenu(void)
{
  // Return a setup menu in case the plugin supports one.
  return new cMenuSetupFreeboxTv;
}

bool cPluginFreeboxTv::SetupParse(const char *Name, const char *Value)
{
  // Parse your own setup parameters and store their values.
  if (!strcasecmp(Name, "ClientPortNum"))
    ClientPortNum = atoi(Value);
  else
    return false;

  return true;
}

const char **cPluginFreeboxTv::SVDRPHelpPages(void)
{
  // Return help text for SVDRP commands this plugin implements
  return NULL;
}

cString cPluginFreeboxTv::SVDRPCommand(const char *Command, const char *Option, int &ReplyCode)
{
  // Process SVDRP commands this plugin implements
  return NULL;
}

VDRPLUGINCREATOR(cPluginFreeboxTv); // Don't touch this!
