/*
 * avards.c: A plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id$
 */

#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <linux/dvb/video.h>
#include <linux/videodev.h>
#include <linux/videodev2.h>
#include <linux/dvb/frontend.h>

#include <vdr/tools.h>
#include <vdr/osdbase.h>

#include "detector.h"
#include "avards.h"

struct SAspectInfo
{
  float       fMinAspect;
  const char* pszName;
  unsigned    iWssData;
  int         Lines;
};

SAspectInfo aAspectInfo[] =
{
  { 0.00,    "unknown",0x08, 576 },
  { 1.46,    "4:3",    0x08, 576 },
  { 1.66,    "L14:9",  0x01, 504 },
  { 1.90,    "L16:9",  0x0b, 432 },
  { FLT_MAX, "L>16:9", 0x0d, 340 },
  { 1.77,    "16:9",   0x07, 576 }
};


cDetector::cDetector(void)
:cThread("avards-detector")
{
  snprintf(pszDvbVideo, MAXSTRING, "%s%s", config.pszDvb, "/video0");
  snprintf(pszDvbFrontend, MAXSTRING, "%s%s", config.pszDvb, "/frontend0");

  lastAspectInfo = 0;
  lastHeight = 0;
}

cDetector::~cDetector()
{
  Stop();
}

#if APIVERSNUM < 10504
bool cDetector::GetMaxOSDSize(avards_MaxOSDsize_v1_0 *OSDsize)
{
  //isyslog("avards: cDetector::GetOSDSize()");
  OSDsize->left   = Setup.OSDLeft;
  OSDsize->width  = Setup.OSDWidth;

  Lock();
  int AspectInfo = lastAspectInfo;
  int Height = lastHeight;
  Unlock();

  if (!Active() || AspectInfo == 0) {
    OSDsize->top    = Setup.OSDTop;
    OSDsize->height = Setup.OSDHeight;
  } else {
    int BlackLines = Height - aAspectInfo[AspectInfo].Lines;
    OSDsize->top    = Setup.OSDTop + BlackLines/2;
    OSDsize->height = max(MINOSDHEIGHT, Setup.OSDHeight - BlackLines);
  }

  return true;
}
#endif

const char *cDetector::GetWSSModeString()
{
  Lock();
  int AspectIndex = lastAspectInfo;
  Unlock();
  return aAspectInfo[AspectIndex].pszName;
}

void cDetector::GetVideoFormat(avards_CurrentVideoFormat_v1_0 *format)
{
  Lock();
  format->ModeString = aAspectInfo[lastAspectInfo].pszName;
  format->Mode       = lastAspectInfo;
  format->Width      = lastWidth;
  format->Height     = lastHeight;
  Unlock();
  return;
}

bool cDetector::StartUp()
{
  if (!Running())
    Start();
  return true;
}

void cDetector::Stop()
{
  if (Running()) {
    Cancel(-1);
    Wait.Signal();
  }
}

void cDetector::Action(void)
{
  naAspectInfo = sizeof(aAspectInfo) / sizeof(SAspectInfo);

  int devDvbVideo = open(pszDvbVideo,O_RDONLY|O_NONBLOCK);
  if (devDvbVideo < 0)
    esyslog("avards: Error: Can't open dvb video device '%s' (%s)\n", pszDvbVideo, strerror(errno));
  else {
    int devDvbFrontend = 0;
    if (config.FrontendHasLockTest)
      devDvbFrontend = open(pszDvbFrontend,O_RDONLY|O_NONBLOCK);
    if (config.FrontendHasLockTest && (devDvbFrontend < 0))
      esyslog("avards: Error: Can't open dvb frontend device '%s' (%s)\n", pszDvbFrontend, strerror(errno));
    else {
      int devVideo = open(config.pszDevVideo, O_RDWR);
      if (devVideo < 0)
        esyslog("avards: Error: Can't open video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
      else {
        wssoverdrive = new cWSSoverdrive;
        if (wssoverdrive) {
          wssoverdrive->OpenVbiDevice(config.pszDevVbi);
          struct video_mbuf mbufVideo;
          if (ioctl(devVideo, VIDIOCGMBUF, &mbufVideo) != 0) 
            esyslog("avards: Error: Can't open video buffer on video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
          else {
            unsigned char *pbVideo = (unsigned char *)mmap(0, mbufVideo.size, PROT_READ | PROT_WRITE, MAP_SHARED, devVideo, 0);
            if (!pbVideo || (pbVideo == (unsigned char *)-1))
              esyslog("avards: Error: Can't map video buffer of video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
            else {
              struct video_capability capabilityVideo;
              if (ioctl(devVideo, VIDIOCGCAP, &capabilityVideo))
                esyslog("avards: Error: Can't query caps of video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
              else {

                struct video_mmap mmapVideo;
                mmapVideo.frame = 0;
                mmapVideo.width = capabilityVideo.maxwidth;
                mmapVideo.height = capabilityVideo.maxheight;
                mmapVideo.format = VIDEO_PALETTE_GREY;

                int iAspectInfo = 0; //-1;
                Lock();
                lastAspectInfo = 0;
                Unlock();
                char ModeBuffer[MAXDELAY];
                memset (&ModeBuffer, 0 /*-1*/, sizeof(ModeBuffer));

                bool bError = false;

                while (!bError && Running()) {
                  bool bFormatChanged = false;
                  fe_status_t FrontendStatus;
                  if (config.FrontendHasLockTest && (bError = (ioctl(devDvbFrontend, FE_READ_STATUS, &FrontendStatus) < 0)))
                    esyslog("avards: Error: Can't get status of device '%s' (%s)\n", pszDvbFrontend, strerror(errno));
                  else {
                    if (!config.FrontendHasLockTest || (FrontendStatus & FE_HAS_LOCK)) {
                      if (config.Mode > MODE_AUTO)
                        iAspectInfo = config.Mode;
                      else {
                        struct video_status statusVideo;
                        bError = ioctl(devDvbVideo, VIDEO_GET_STATUS, &statusVideo) < 0;
                        if (bError)
                          esyslog("avards: Error: Can't get status of device '%s' (%s)\n", pszDvbVideo, strerror(errno));
                        else {
                          switch (statusVideo.video_format) {
                          case VIDEO_FORMAT_16_9:
                          case VIDEO_FORMAT_221_1:
                            iAspectInfo = 5;
                            break;
                          case  VIDEO_FORMAT_4_3:
                          default:
                            if (iAspectInfo == 5)
                              iAspectInfo = 0; //-1;
                            // grab picture
                            bError = (ioctl(devVideo, VIDIOCMCAPTURE, &mmapVideo) | ioctl(devVideo, VIDIOCSYNC, &mmapVideo.frame));
                            if (bError)
                              esyslog("avards: Error: Can't grab on video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
                            else {
                              const int nMaxLogoWidth = int(mmapVideo.width*config.MaxLogoWidth_pct/100);
                              const int nYoverscan = int (mmapVideo.height*config.Overscan_pct/100);
                              const int nXoverscan = int (mmapVideo.width*config.Overscan_pct/100);

                              unsigned char iPanToneMin;
                              unsigned char iPanToneMax;

                              int nActiveLines = 0;
                              int iBlackLines = 999;
                              int iYtop = nYoverscan;

                              if (config.PanToneBlackMax) {
                                iPanToneMin = 0;
                                iPanToneMax = config.PanToneBlackMax;
                              }
                              else {
                                int aHistogram[256];
                                memset(aHistogram, 0, sizeof(aHistogram));
                                unsigned char *pbVideoLine = pbVideo+iYtop*mmapVideo.width;
                                for (int iX = nXoverscan; iX < mmapVideo.width-nXoverscan; iX++)
                                  aHistogram[pbVideoLine[iX]]++;
                                int iXmax = 0;
                                for (int i = 0; i < 256; i++)
                                  if (aHistogram[i] > aHistogram[iXmax])
                                    iXmax = i;
                                  if (iXmax - config.PanToneTolerance < 0)
                                    iPanToneMin = 0;
                                  else
                                    iPanToneMin = iXmax - config.PanToneTolerance;
                                  if (iXmax + config.PanToneTolerance > 255)
                                    iPanToneMax = 255;
                                  else
                                    iPanToneMax = iXmax + config.PanToneTolerance;
                              }

                              while (iYtop < mmapVideo.height/2) {
                                unsigned char *pbVideoLine = pbVideo+iYtop*mmapVideo.width;
                                int iXleft = nXoverscan;
                                while ( (iXleft < (mmapVideo.width-nXoverscan)) &&
                                        (pbVideoLine[iXleft] >= iPanToneMin)    &&
                                        (pbVideoLine[iXleft] <= iPanToneMax)  )
                                  iXleft++;

                                int iXright = mmapVideo.width-nXoverscan-1;
                                while ( (iXright > iXleft) &&
                                        (pbVideoLine[iXright] >= iPanToneMin) &&
                                        (pbVideoLine[iXright] <= iPanToneMax) )
                                  iXright--;

                                if (iXright - iXleft > nMaxLogoWidth)
                                  break;
                                iYtop++;
                              }

                              if (iYtop < mmapVideo.height/2) {
                                int iYbottom = nYoverscan;
                                while (iYbottom < mmapVideo.height/2)
                                {
                                  unsigned char *pbVideoLine = pbVideo+(mmapVideo.height-1-iYbottom)*mmapVideo.width;
                                  int iXleft = nXoverscan;
                                  while ( (iXleft < (mmapVideo.width-nXoverscan)) &&
                                          (pbVideoLine[iXleft] >= iPanToneMin) &&
                                          (pbVideoLine[iXleft] <= iPanToneMax)  )
                                    iXleft++;

                                  int iXright = mmapVideo.width-nXoverscan-1;
                                  while ( (iXright > iXleft) &&
                                          (pbVideoLine[iXright] >= iPanToneMin) &&
                                          (pbVideoLine[iXright] <= iPanToneMax) )
                                    iXright--;

                                  if (iXright - iXleft > nMaxLogoWidth)
                                    break;
                                  iYbottom++;
                                }

                                if (iYbottom < mmapVideo.height/2) {
                                  iBlackLines = 2*min(iYtop, iYbottom);
                                  nActiveLines = mmapVideo.height - iBlackLines;
                                }
                              }

                              if (nActiveLines > 0) {
                                const float fLinearWidth = 4./3.*mmapVideo.height;
                                if (iAspectInfo == 0)
                                  bFormatChanged = true;
                                else {
                                  const int nAspectHyst = 2;
                                  int nMinActiveLines = (int)(fLinearWidth/aAspectInfo[iAspectInfo].fMinAspect) - nAspectHyst;
                                  int nMaxActiveLines =   (iAspectInfo > 1) //(iAspectInfo > 0)
                                    ? (int)(fLinearWidth/aAspectInfo[iAspectInfo-1].fMinAspect) + nAspectHyst
                                    : mmapVideo.height;
                                  bFormatChanged = (nActiveLines < nMinActiveLines) || (nActiveLines > nMaxActiveLines);
                                }
                                float fAspect = fLinearWidth/nActiveLines;
                                if (bFormatChanged) {
                                  iAspectInfo = 1; //0;
                                  while (fAspect > aAspectInfo[iAspectInfo].fMinAspect && iAspectInfo < (config.Overzoom?4:3))
                                    iAspectInfo++;

                                  int iLinesIndex = config.Overzoom ? 4 : 3;
                                  while (nActiveLines > aAspectInfo[iLinesIndex].Lines && iLinesIndex > 1)
                                    iLinesIndex--;
                                  if (iAspectInfo != iLinesIndex)
                                    //isyslog("avards Delta: fAspect=%.2f=%6s(%d/%d) ActiveLines=%3d", fAspect, aAspectInfo[iAspectInfo].pszName, iAspectInfo, iLinesIndex, nActiveLines);
                                    isyslog("avards Delta: fAspect=%.3f =>%6s %.2f(%d)/%.2f(%d) ActiveLines=%3d/%3d",
                                            fAspect, aAspectInfo[iAspectInfo].pszName, aAspectInfo[iAspectInfo].fMinAspect, iAspectInfo, aAspectInfo[iLinesIndex].fMinAspect, iLinesIndex, nActiveLines, mmapVideo.height);
#ifdef AVARDS_SHOW_ALL_LINES
                                  //iAspectInfo = iLinesIndex;
#endif
                                }
                              }
                            }
                            break;
                          }
                        }
                      }
                    }
                  }

                  if (!bError) {
                    int iDelay = 0;
                    if (iAspectInfo == 5)
                      iDelay = config.Delay;
                    else
                      if (iAspectInfo >0)
                      { // 4:3 delay
                        while ((iDelay < config.Delay) && ModeBuffer[iDelay] == iAspectInfo)
                          iDelay++;
                      }

                    if (iDelay == config.Delay && lastAspectInfo != iAspectInfo) {
                      // write WSS data to device
                      isyslog("avards: switching to %s(%d)", aAspectInfo[iAspectInfo].pszName, iAspectInfo);
                      wssoverdrive->SetWSS(aAspectInfo[iAspectInfo].iWssData);
#if APIVERSNUM >= 10504
                      int BlackLines = mmapVideo.height - aAspectInfo[iAspectInfo].Lines;
                      cOsd::SetOsdPosition(Setup.OSDLeft,
                                           iAspectInfo?Setup.OSDTop + BlackLines/2 : Setup.OSDTop,
                                           Setup.OSDWidth,
                                           iAspectInfo?Setup.OSDHeight - BlackLines : Setup.OSDHeight);
#endif
                      Lock();
                      lastAspectInfo = iAspectInfo;
                      lastHeight = mmapVideo.height;
                      lastWidth = mmapVideo.width;
                      Unlock();
                    }

                    if (config.Delay) {
                      memmove(&ModeBuffer[1], &ModeBuffer[0], config.Delay-1);
                      ModeBuffer[0] = iAspectInfo;
                    }
                  }

                  if (Running())
                    Wait.Wait(max(80, config.PollRate));
                }

              }
              munmap(pbVideo, mbufVideo.size);
            }
          }
        }
#if APIVERSNUM >= 10504
        cOsd::SetOsdPosition(Setup.OSDLeft, Setup.OSDTop, Setup.OSDWidth, Setup.OSDHeight);
#endif
        wssoverdrive->CloseVbiDevice();
        delete wssoverdrive;
        wssoverdrive = NULL;
        close(devVideo);
      }
      if (config.FrontendHasLockTest)
        close(devDvbFrontend);
    }
    close(devDvbVideo);
  }
}


cWSSoverdrive::cWSSoverdrive(void)
{
  vbi_device = 0;
  lastWssData = -1;
}

cWSSoverdrive::~cWSSoverdrive(void)
{
  if (vbi_device)
    CloseVbiDevice();
}

bool cWSSoverdrive::OpenVbiDevice(const char* szDevice)
{
  vbi_device = open(szDevice, O_WRONLY);
  if (vbi_device < 0) 
  {
    esyslog("avards: Error: Can't open vbi device '%s' (%s)\n", szDevice, strerror(errno));
    return false;
  }

  struct v4l2_format fmt;
  memset(&fmt, 0, sizeof fmt);
  fmt.type = V4L2_BUF_TYPE_SLICED_VBI_OUTPUT;
  fmt.fmt.sliced.service_set = V4L2_SLICED_WSS_625;
  if (ioctl(vbi_device, VIDIOC_S_FMT, &fmt) < 0) {
    esyslog("avards Error: Can't init vbi device '%s' (%s)\n", szDevice, strerror(errno));
    CloseVbiDevice();
    return false;
  }

  return true;
}

void cWSSoverdrive::CloseVbiDevice(void)
{
  if (vbi_device) {
    close(vbi_device);
    vbi_device = 0;
  }
}

void cWSSoverdrive::SetWSS(int iWssData)
{
  if (vbi_device && iWssData != lastWssData) {
    isyslog("avards: setting WSS data to %02x", iWssData);
    struct v4l2_sliced_vbi_data data;
    memset(&data, 0, sizeof data);
    data.id = (iWssData) ? V4L2_SLICED_WSS_625 : 0;
    data.line = 23;
    data.data[0] = iWssData & 0xff;
    data.data[1] = (iWssData >> 8) & 0x3f;
    if (write(vbi_device, &data, sizeof data) != sizeof data) {
      esyslog("avards: Error: Can't write to device (%s)\n", strerror(errno));
      CloseVbiDevice();
    }
    lastWssData = iWssData;
  }
}
