
#include <string>
#include <fstream>

#include <vdr/interface.h>
#include <vdr/menu.h>
#include <vdr/plugin.h>

#ifdef HAVE_ICONPATCH
#include <vdr/iconpatch.h>
#endif

#include "commands.h"
#include "i18n.h"
#include "player-mp3.h"
#include "data-mp3.h"
#include "decoder.h"
#include "mp3control.h"
#include "skin.h"

#define X_DISPLAY_MISSING
#define MAXLENGTH 256
#define THEMESEXT "*.theme"
#define FINDCMD "cd '%s' && find '%s' -iname '%s' -printf '%%p\n' | sort -f"

char urlname[256];
bool recordstream, deletetrack, deletetracks;


// ------------ cMP3SearchID3Info --------------------------------------------------------------------------------------------------- //

class cMP3SearchID3Info : public cOsdMenu {
private:
  cOsdItem *Item(const char *name, const char *text);
  cOsdItem *Item(const char *name, const char *format, const float num);
  void Build(cSongInfo *info, const char *name, const char *path);
public:
  cMP3SearchID3Info(cSong *song);
  cMP3SearchID3Info(cSongInfo *si, const char *name, const char *path);
  virtual eOSState ProcessKey(eKeys Key);
};


cMP3SearchID3Info::cMP3SearchID3Info(cSong *song)
:cOsdMenu(tr("ID3 information"),16)
{
  Build(song->Info(),song->Name(),song->SubPath());
}

cMP3SearchID3Info::cMP3SearchID3Info(cSongInfo *si, const char *name, const char *path)
:cOsdMenu(tr("ID3 information"),16)
{
  Build(si,name,path);
}

void cMP3SearchID3Info::Build(cSongInfo *si, const char *name, const char *path)
{
  if(si) {
//    Item(tr("Trackinfo:"),"");
    Item(tr("Location"),path);
    if(si->HasInfo() && si->Total>0) {
      char *buf1=0;
      char *buf2=0;
      int totaltime = mgr->GetListLength();
      asprintf(&buf1,"%02d:%02d",si->Total/60,si->Total%60);
      asprintf(&buf2,"%02d:%02d", totaltime/60, totaltime%60);
      Item(tr("Length"),buf1);
      free(buf1);
      Item(tr("Title"),si->Title);
      Item(tr("Artist"),si->Artist);
      Item(tr("Album"),si->Album);
      Item(tr("Year"),0,(float)si->Year);
      Item(tr("Genre"),si->Genre);
      Item(tr("Rating"),0,si->Rating);
      Item(tr("Samplerate"),"%.1f kHz",si->SampleFreq/1000.0);
      Item(tr("Bitrate"),"%.f kbit/s",si->Bitrate/1000.0);
      Item(tr("Channels"),0,(float)si->Channels);
      Item(tr("Comment"),si->Comment);
      Item(tr("Playlist time"), buf2);
      Item(tr("Playlist tracks"),0,(float)(mgr->maxIndex +1));
      free(buf2);
      }



    Display();
    }
}

cOsdItem *cMP3SearchID3Info::Item(const char *name, const char *format, const float num)
{
  cOsdItem *item;
  if(num>=0.0) {
    char *buf=0;
    asprintf(&buf,format?format:"%.f",num);
    item=Item(name,buf);
    free(buf);
    }
  else item=Item(name,"");
  return item;
}

cOsdItem *cMP3SearchID3Info::Item(const char *name, const char *text)
{
  char *buf=0;
  asprintf(&buf,"%s\t%s",name,text?text:"");
  cOsdItem *item = new cOsdItem(buf,osBack);
  item->SetSelectable(false);
  free(buf);
  Add(item); return item;
}


eOSState cMP3SearchID3Info::ProcessKey(eKeys Key)
{
  eOSState state = cOsdMenu::ProcessKey(Key);

  if(state==osUnknown) {
     switch(Key) {
       case kRed:
       case kGreen:
       case kYellow:
       case kOk:
       case kBlue:   return osBack;
       default: return osContinue;;
       }
     }
  return state;
}



//---------------- cMP3UpdateWebStreams --------------------------------------------------------------------------------------------------------------------------


class cMP3UpdateWebStreams : public cOsdMenu {
private:
  cCommands commands;
//  cOsdItem *Item(const char *name, const char *text);
  void SetHelp();
  eOSState Execute(void);
public:
  cMP3UpdateWebStreams(void);
  void LoadCommand();
  virtual eOSState ProcessKey(eKeys Key);
};

cMP3UpdateWebStreams::cMP3UpdateWebStreams(void)
:cOsdMenu(tr("MP3ng Update WebStreams"))
{
  SetHelp();

  LoadCommand();

  for (cCommand *command = commands.First(); command; command = commands.Next(command)) {
    cOsdItem *item = new cOsdItem(hk(command->Title()));
    Add(item, osUnknown);
    if(strstr(item->Text(), "----------------")) 
      item->SetSelectable(false);
    else
      item->SetSelectable(true);
  }
}


void cMP3UpdateWebStreams::SetHelp() {
  cOsdMenu::SetHelp(tr("Parent"),NULL,NULL,NULL);
}


void cMP3UpdateWebStreams::LoadCommand() {

    commands.Load(AddDirectory(cPlugin::ConfigDirectory(i18n_name), "/data/webstreams.dat"), true);
}


eOSState cMP3UpdateWebStreams::Execute(void) {

// Execute usercommands;;
    cCommand *command = commands.Get(Current());
    if (command) {
      char *buffer = NULL;
      bool confirmed = true;

      if (command->Confirm()) {
        asprintf(&buffer, "%s?", command->Title());
        confirmed = Interface->Confirm(buffer);
        free(buffer);
      }
      if (confirmed) {
        asprintf(&buffer, "%s...", command->Title());
        Skins.Message(mtStatus, buffer);
        free(buffer);
         
        const char *Result = command->Execute(false);

        Skins.Message(mtStatus, NULL);
        if(Result) return AddSubMenu(new cMenuText(command->Title(), Result, fontFix));
	return osBack;
      }
    }

  return osContinue;
}


eOSState cMP3UpdateWebStreams::ProcessKey(eKeys Key) {

  bool hadSubmenu = HasSubMenu();

  eOSState state = cOsdMenu::ProcessKey(Key);

  if (hadSubmenu && !HasSubMenu())
    return osBack;

  if (state == osUnknown) {
    switch (Key) {
        case kRed:   return osBack;
	case kGreen:
	case kYellow:
	case kBlue:  return osContinue;
      case kOk:      return Execute(); 
      default:       break;
    }
  }
  
  return state;
}        		



// ------------ cMP3ThemesItem ----------------------------------------------------------------------------------------------- //

class cMP3ThemesItem : public cOsdItem {
private:
  cFileObjItem *item;
  virtual void Set(void);
  int idx;
public:
  cMP3ThemesItem(cFileObjItem *Item, int index);
  cFileObjItem *Item(void) {return item;}
  ~cMP3ThemesItem();
};


cMP3ThemesItem::cMP3ThemesItem(cFileObjItem *Item, int index) {
  idx = index; 
  item = Item;
  Set();  
}

cMP3ThemesItem::~cMP3ThemesItem(){
}

void cMP3ThemesItem::Set(void) {
  char *Name;
//  asprintf(&Name,"%02d.\t%s", idx , item->Name());
  asprintf(&Name,"%s", item->Name());
  SetText(Name,false);
}


// ------------ cMP3Themes --------------------------------------------------------------------------------------------------- //

class cMP3Themes : public cOsdMenu {
private:
  void SetHelp();
  void LoadSkinFiles();
  eOSState SetTheme(void);
  cFileObjItem *newitem;
protected:
  cFileObjItem *CurrentFileObjItem(void);  
public:
  cMP3Themes(void);
  virtual eOSState ProcessKey(eKeys Key);
};


cMP3Themes::cMP3Themes(void)
:cOsdMenu(tr("MP3ng Colorthemes"),4)
{
  LoadSkinFiles();
  SetHelp();
}

void cMP3Themes::SetHelp() {
  cOsdMenu::SetHelp(tr("Parent"),NULL,NULL,NULL);
}


void cMP3Themes::LoadSkinFiles()
{
  int count = 0;

  char *cmd = NULL;
  cReadLine reader;

  std::string plugindir;
  std::string themename;
  std::string themefile;
  std::string Base;
   
  plugindir = cPlugin::ConfigDirectory(i18n_name);
  plugindir = plugindir + "/themes/";

  int  len2 = plugindir.length()+1;

   
  asprintf(&cmd, FINDCMD, plugindir.c_str(), plugindir.c_str(), THEMESEXT);
  FILE *p = popen(cmd, "r");

  if(p) {
    char *s;
    while ((s = reader.Read(p)) != NULL) {
      count++;
      themefile = s;
      int len = themefile.length();
      std::string::size_type pos = themefile.rfind('/',len);
      if(pos!=std::string::npos) {
        // Filename
        themename = themefile.substr(pos+1,len);
        // Basedir
        int sublen = len - len2 - themename.length()-1;
        Base = themefile.substr(len2,sublen);
      }
      newitem = new cFileObjItem(plugindir.c_str(), themename.c_str(), otFile);
      if(newitem && newitem->Type()==otFile) {
        Add(new cMP3ThemesItem(newitem, count));
      }
    }
    pclose(p);
  }
  else
    Skins.Message(mtError, "ERROR: Cant find any themes!");
    
  free(cmd);
}          
      
cFileObjItem *cMP3Themes::CurrentFileObjItem(void) {
  cMP3ThemesItem *item = (cMP3ThemesItem *)Get(Current());
  return item ? item->Item():0;
}  

  
eOSState cMP3Themes::SetTheme(void) {
  cFileObjItem *item=CurrentFileObjItem();

  if(item && item->Type()==otFile) {
    if(MP3Skin.StoreSkin(item->Name())) {
      MP3Skin.ParseSkin("current.colors", true);
      return osBack;
      }
    else    
      return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not store/load new themefile !"), fontFix));
  }
  
  return osContinue;
}


eOSState cMP3Themes::ProcessKey(eKeys Key)
{
  eOSState state = cOsdMenu::ProcessKey(Key);

  if(state==osUnknown) {
     switch(Key) {
       case kOk:
          state = SetTheme();
	  break;     
       case kBlue:
       case kGreen:
       case kYellow:
       case kRed: return osBack;
       default: return osContinue;;
       }
     }
  return state;
}

// ------------ cMP3EditPlaylist ---------------------------------------------------------------------------------------------- //

cMP3EditPlaylist::cMP3EditPlaylist(void)
:cOsdMenu(tr("Edit playlist..."))
{
//0
  Add(new cOsdItem(hk(tr("Delete track from playlist only"))));
//1
  Add(new cOsdItem(hk(tr("Delete selected file from medium"))));
//2
  Add(new cOsdItem(hk(tr("Delete everything in playlist from medium only"))));
//3
  Add(new cOsdItem(hk(tr("Empty playlist except first track"))));
//4
  Add(new cOsdItem(hk(tr("Save playlist for later burning"))));

  SetHelp();
}


void cMP3EditPlaylist::SetHelp() {
  cOsdMenu::SetHelp(tr("Parent"),NULL,NULL,NULL);
}


eOSState cMP3EditPlaylist::Execute(void) {

  int ItemIndex = Current();
  
//0/ Delete Track from playlist and save as current playlist in ConfigDir;;
    if (ItemIndex == 0) {
      if(CannotDelete) {
        Skins.Message(mtError, tr("You can't delete stream"));
        return osContinue;
      }
      
      if(MP3Setup.AdminMode) {
        if (!mgr->maxIndex==0 && !mgr->ShuffleMode()) {
          deletetrack=true;
          return osBack;
	  }
	else
	  return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not delete track !"), fontFix));
	}
      else {	
        if(Interface->Confirm(tr("Delete track from playlist ?"))  && Interface->Confirm(tr("Are you sure?"))  ) {
          if (!mgr->maxIndex==0 && !mgr->ShuffleMode())  {
            deletetrack=true;
	    return osBack;
	    }
          else
            return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not delete track !"), fontFix));
        }
      }
    }      

      
//1/ Delete file from medium
    if (ItemIndex == 1) {
      if(CannotDelete) {
        Skins.Message(mtError, tr("You can't delete stream"));
	return osContinue;
      }

      if(MP3Setup.AdminMode) {
        if (!mgr->maxIndex==0 && !mgr->ShuffleMode()) {
          RemoveFileOrDir(Songname, false);
          deletetrack=true;
          return osBack;
	  }
	else
	  return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not delete track !"), fontFix));
	}
      else {	
        if(Interface->Confirm(tr("Delete file from medium ?"))  && Interface->Confirm(tr("Are you sure?"))  ) {
          if (!mgr->maxIndex==0 && !mgr->ShuffleMode()) {
            RemoveFileOrDir(Songname, false);
            deletetrack=true;
            return osBack;
	    }
	  else
  	    return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not delete track !"), fontFix));
        }
      }
    }



//2/ Delete everything in playlist from medium
    if (ItemIndex == 2) {
      if(CannotDelete) {
        Skins.Message(mtError, tr("You can't delete stream"));
	return osContinue;
      }

      if(Interface->Confirm(tr("Delete everything in playlist from medium ?"))  && Interface->Confirm(tr("Are you sure?"))  ) {
        Skins.Message(mtStatus, tr("Delete all files from medium..."));
        bool Result = DeleteList();
        Skins.Message(mtStatus, NULL);
        if (Result) 
          return AddSubMenu(new cMenuText(tr("Delete"), tr("Files terminated !"), fontFix));
	else	  
	  return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not delete tracks !"), fontFix));
      }
    }
 

//3/ Empty playlist except first track;;
    if (ItemIndex == 3) {
      if(CannotDelete) {
        Skins.Message(mtError, tr("You can't delete stream"));
	return osContinue;
      }
      
      if(MP3Setup.AdminMode) {
        if (!mgr->maxIndex==0 && !mgr->ShuffleMode()) {
	  deletetracks=true;
          return osBack;
	  }
	else  
	  return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not delete tracks !"), fontFix));
	}
      else {	
        if(Interface->Confirm(tr("Empty playlist except first track ?")) ) {
          if (!mgr->maxIndex==0 && !mgr->ShuffleMode()) {
            Skins.Message(mtStatus, tr("Delete tracks from playlist..."));
            deletetracks=true;
            Skins.Message(mtStatus, NULL);
	    return osBack;
	    }
	  else
	    return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not delete tracks !"), fontFix));
        }
      }
    }      


//4/ Save playlist for burn..something with full path in ConfigDir;;
    if (ItemIndex == 4) {
      if(MP3Setup.AdminMode) {
        Skins.Message(mtStatus, tr("Save playlist for burning..."));
        char *buf=0;
        asprintf(&buf, "%s/%s", cPlugin::ConfigDirectory(i18n_name), "playlists/burnlist.m3u");
        bool Result = mgr->SaveList(buf, true);
        Skins.Message(mtStatus, NULL);
        free(buf);
        if (Result) 
          return AddSubMenu(new cMenuText(tr("Save burnlist"), tr("Playlist saved as burnlist.m3u !"), fontFix));
        else 
          return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not save burnlist burnlist.m3u !"), fontFix));
        }
      else {
        if(Interface->Confirm(tr("Overwrite last burnlist for later burnprocess ?"))  ) {
          Skins.Message(mtStatus, tr("Save playlist for burning..."));
          char *buf=0;
          asprintf(&buf, "%s/%s", cPlugin::ConfigDirectory(i18n_name), "playlists/burnlist.m3u");
          bool Result = mgr->SaveList(buf, true);
          Skins.Message(mtStatus, NULL);
          free(buf);
          if (Result) 
            return AddSubMenu(new cMenuText(tr("Save burnlist"), tr("Playlist saved as burnlist.m3u !"), fontFix));
          else 
            return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not save burnlist burnlist.m3u !"), fontFix));
        }
      }
    }

  return osContinue;
}

eOSState cMP3EditPlaylist::ProcessKey(eKeys Key) {

  bool hadSubmenu = HasSubMenu();

  eOSState state = cOsdMenu::ProcessKey(Key);

  if (hadSubmenu && !HasSubMenu())
    return osBack;

  if (state == osUnknown) {
    switch (Key) {
        case kRed:
	  return osBack;
	break;  
      case kOk:
          return Execute();
	break;  
      default:
           state = osContinue;
    }
  }
  
  return state;
}        		


bool cMP3EditPlaylist::DeleteList(void) {
  std::ifstream filestr;
  std::string datei;
  std::string line;
  int count=0;
  bool res = false;

  datei=cPlugin::ConfigDirectory(i18n_name);
  datei=datei + "/playlists/deletelist.m3u";

  if(mgr->SaveList(datei.c_str(), true)) {
    filestr.open (datei.c_str());
    while ((getline(filestr,line,'\n'))) {
      count++;
      isyslog("mp3ng: Delete File : %i  %s\n",count,line.c_str());
      RemoveFileOrDir(line.c_str(), false);
    }
    filestr.close();
    res = true;
  }

  return res;
} 


// ------------ cMP3Commands --------------------------------------------------------------------------------------------------- //

cMP3Commands::cMP3Commands(void)
:cOsdMenu(tr("MP3ng Commands"))
{
/* 0*/  Add(new cOsdItem(hk(tr("Edit playlist..."))));
/* 1*/  Add(new cOsdItem(hk(tr("Search in categories..."))));
/* 2*/  Add(new cOsdItem(hk(tr("Show ID3 information of song"))));
/* 3*/  Add(new cOsdItem(hk(tr("Update webstreams..."))));
/* 4*/  if(!recordstream)
          Add(new cOsdItem(hk(tr("Start Recording"))));
        else
          Add(new cOsdItem(hk(tr("Stop Recording"))));
/* 5*/  if(!mgr->ShuffleMode())
          Add(new cOsdItem(hk(tr("Shufflemode on"))));
        else    
          Add(new cOsdItem(hk(tr("Shufflemode off"))));
/* 6*/  if(!mgr->LoopMode())
          Add(new cOsdItem(hk(tr("Loopmode on"))));
        else    
          Add(new cOsdItem(hk(tr("Loopmode off"))));
/* 7*/  Add(new cOsdItem(hk(tr("Load new colortheme..."))));
/* 8*/ // Add(new cOsdItem(hk(tr("Rating"))));

  SetHelp();
  LoadCommands();

  for (cCommand *command = commands.First(); command; command = commands.Next(command)) {
    cOsdItem *item = new cOsdItem(hk(command->Title()));
    Add(item, osUnknown);
    if(strstr(item->Text(), "----------")) 
      item->SetSelectable(false);
    else
      item->SetSelectable(true);
  }

    Display();
}


cMP3Commands::~cMP3Commands()
{
}

void cMP3Commands::SetHelp() {
  cOsdMenu::SetHelp(tr("Parent"),NULL,NULL,NULL);
}

void cMP3Commands::LoadCommands() {

    commands.Load(AddDirectory(cPlugin::ConfigDirectory(i18n_name), "/data/mp3ngcmds.dat"), true);
}

eOSState cMP3Commands::Execute(void) {

  int ItemIndex = Current();
    
//0/ Edit Playlist;;
    if (ItemIndex == 0) {
      return AddSubMenu(new cMP3EditPlaylist());
    }


//1/ Search;;
    if (ItemIndex == 1) {
      return AddSubMenu(new cMP3Search());
    }


//2/ Show ID3Info;;
    if (ItemIndex == 2) {
      cSong *song = new cSong(mgr->curr);
      if(song) {
        return AddSubMenu(new cMP3SearchID3Info(song));
	}
      else {
          return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not get songinfo !"), fontFix));
      }
    }	
    

//3/ Update Webstreams;;
    if (ItemIndex == 3) {
      return AddSubMenu(new cMP3UpdateWebStreams());
    }


//4/ Start<->Stop Recording;;
    if (ItemIndex == 4) {

        if (MP3Setup.AdminMode) {
          if (recordstream) {
            StopRecord();
            return AddSubMenu(new cMenuText(tr("Stop Recording"), tr("Recording stopped !"), fontFix));
            }
          else {
            if (isWebstream) {
              StartRecord();
              return AddSubMenu(new cMenuText(tr("Start Recording"), tr("Recording started !"), fontFix));
            }
          }
        }
        else {	
          if(recordstream) {
            if(Interface->Confirm(tr("Stop recording ?"))  ) {
              StopRecord();
              return AddSubMenu(new cMenuText(tr("Stop Recording"), tr("Recording stopped !"), fontFix));
            }
	  }    
          else {
            if (isWebstream) {
              if(Interface->Confirm(tr("Start recording ?"))  ) {
                StartRecord();
                return AddSubMenu(new cMenuText(tr("Start Recording"), tr("Recording started !"), fontFix));
              }
            }
          }
        }
    }


//5/ Shufflemode;;
    if (ItemIndex == 5) {
      mgr->ToggleShuffle();
      return osBack;
    }  


//6/ Loopmode;;
    if (ItemIndex == 6) {
      mgr->ToggleLoop();
      return osBack;
    }  


//7/ Themes;;
    if (ItemIndex == 7) {
      return AddSubMenu(new cMP3Themes());
    }


//8/ Evaluate track;;
//    if (ItemIndex == 8) {
//      return AddSubMenu(new cMP3Rating());
//    }


// From NOW on execute usercommands;;
  if (ItemIndex >= 8) {
    cCommand *command = commands.Get(ItemIndex-8);
    if (command) {
      char *buffer = NULL;
      bool confirmed = true;

      std::string Artist ="";
      std::string Album  =""; 
      std::string Coverdir ="";
    
      cSongInfo *si = mgr->Current()->Info(false);
      if(!si) si = mgr->Current()->Info();
      if(si && si->Artist)Artist = si->Artist;
      if(si && si->Album) Album  = si->Album;
      if (command->Confirm()) {
        asprintf(&buffer, "%s?", command->Title());
        confirmed = Interface->Confirm(buffer);
        free(buffer);
      }
      if (confirmed) {
        asprintf(&buffer, "%s...", command->Title());
        Skins.Message(mtStatus, buffer);
        free(buffer);

        asprintf(&buffer, "\"%s\"  \"%s\"  \"%s\"  \"%s\"  \"%s\"", Songname, MP3Setup.CopyDir, Artist.c_str(), Album.c_str(), MP3Setup.CoverDir);
        const char *Result = command->Execute(buffer);

        d(printf("Executed: %s\n", buffer));
	isyslog("Executed: %s\n", buffer);

        free(buffer);
        Skins.Message(mtStatus, NULL);
        if(Result) return AddSubMenu(new cMenuText(command->Title(), Result, fontFix));
	return osBack;
      }
    }
  }

  return osContinue;
}

// ------------------ STARTRECORD -----------------------//
void cMP3Commands::StartRecord(void) {
 FILE *recordcmd;
 char *buffer;

  d(printf("MP3ng: Started Recording\n"));
  asprintf(&buffer, "streamripper %s -d '%s' -w %s/data/parse_rules.txt %s &", urlname, MP3Setup.RecordDir, cPlugin::ConfigDirectory(i18n_name),MP3Setup.RecordOpts);
  recordcmd = popen(buffer, "r");
  d(printf("MP3ng: Executed '%s'\n", buffer));
  pclose(recordcmd);
  
  free(buffer);
  recordstream=true;
}

// ------------------ STOPRECORD -----------------------//
void cMP3Commands::StopRecord(void) {
 FILE *stoprecordcmd;
 char *buffer;

  d(printf("MP3ng: Stopped Recording\n"));
  asprintf(&buffer, "killall -15 streamripper");
  stoprecordcmd = popen(buffer, "r");
  d(printf("MP3ng: Executed '%s'\n", buffer));
  pclose(stoprecordcmd);
      
  free(buffer);
  recordstream=false;
}

eOSState cMP3Commands::ProcessKey(eKeys Key) {

  bool hadSubmenu = HasSubMenu();

  eOSState state = cOsdMenu::ProcessKey(Key);

  if (hadSubmenu && !HasSubMenu())
    return osBack;

  if (state == osUnknown) {
    switch (Key) {
        case kRed:
	  return osBack;
	break;  
      case kOk:
          return Execute();
	break;  
      default:
           state = osContinue;
    }
  }
  
  return state;
}        		


// ------------ cMP3SearchBrowseItem ------------------------------------------------------------------------------------------------ //

class cMP3SearchBrowseItem : public cOsdItem {
private:
 cFileObj *item;
 virtual void SetID3(void);
 int idx;
public:
 cMP3SearchBrowseItem(cFileObj *Item, int index);
 cFileObj *Item(void) {return item;}
 void NewName(void);
};

cMP3SearchBrowseItem::cMP3SearchBrowseItem(cFileObj *Item, int index) {
  idx = index;
  item = Item;
  SetID3();
}

/*
void cMP3SearchBrowseItem::SetID3(void) {
  char *Name;
  cSong *song = new cSong(item);
  cSongInfo *si=song->Info(false);
  if(!si) si=song->Info();

  if(MP3Setup.ShowFileName){
    asprintf(&Name,"%04d. \t <%s> \t               \t         ",idx, song->Name());
    }
  else {    
    if(si && si->Title) {
      if(MP3Setup.ArtistFirst) 
        asprintf(&Name,"%04d. \t %s - %s \t %3d k    \t%02d:%02d",idx, si->Artist, si->Title, si->Bitrate/1000, si->Total/60, si->Total%60);
      else 
        asprintf(&Name,"%04d. \t %s - %s \t %3d k    \t%02d:%02d",idx, si->Title, si->Artist, si->Bitrate/1000, si->Total/60, si->Total%60);

    }
    else {
      asprintf(&Name,"%04d. \t <%s> \t               \t         ",idx, song->Name());
    }
  }

  SetText(Name,false);
}
*/

void cMP3SearchBrowseItem::SetID3(void) {
  char *Name;
  cSong *song = new cSong(item);
//  cSongInfo *si=song->Info(false);
//  if(!si) si=song->Info();

  if(MP3Setup.ShowFileName){
    asprintf(&Name,"%04d. \t <%s> \t               \t         ",idx, song->Name());
    }
/*  else {    
    if(si && si->Title) {
      if(MP3Setup.ArtistFirst) 
        asprintf(&Name,"%04d. \t %s - %s \t %3d k    \t%02d:%02d",idx, si->Artist, si->Title, si->Bitrate/1000, si->Total/60, si->Total%60);
      else 
        asprintf(&Name,"%04d. \t %s - %s \t %3d k    \t%02d:%02d",idx, si->Title, si->Artist, si->Bitrate/1000, si->Total/60, si->Total%60);

    }
    else {
      asprintf(&Name,"%04d. \t <%s> \t               \t         ",idx, song->Name());
    }
  }
*/
  SetText(Name,false);
  delete song;
}




void cMP3SearchBrowseItem::NewName(void) {
  char *Name;
  cSong *song = new cSong(item);
  cSongInfo *si=song->Info(false);
  if(!si) si=song->Info();

#ifdef HAVE_ICONPATCH
  if(si && si->Title) {
    if(MP3Setup.ArtistFirst) 
      asprintf(&Name,"%c \t %s - %s \t %3d k    \t%02d:%02d", ICON_PFEIL, si->Artist, si->Title, si->Bitrate/1000, si->Total/60, si->Total%60);
    else 
      asprintf(&Name,"%c. \t %s - %s \t %3d k    \t%02d:%02d",ICON_PFEIL, si->Title, si->Artist, si->Bitrate/1000, si->Total/60, si->Total%60);

  }
  else {
    asprintf(&Name,"%c \t <%s> \t               \t         ",ICON_PFEIL, song->Name());
  }
#else
  if(si && si->Title) {
    if(MP3Setup.ArtistFirst) 
      asprintf(&Name,"%s \t %s - %s \t %3d k    \t%02d:%02d", "<<  .", si->Artist, si->Title, si->Bitrate/1000, si->Total/60, si->Total%60);
    else 
      asprintf(&Name,"%s. \t %s - %s \t %3d k    \t%02d:%02d", "<<  .", si->Title, si->Artist, si->Bitrate/1000, si->Total/60, si->Total%60);

  }
  else {
    asprintf(&Name,"%s \t <%s> \t               \t         ", "<<  .", song->Name());
  }
#endif
 
  SetText(Name,false);
}



// ------------ cMP3SearchResult ------------------------------------------------------------------------------------------------------------

cFileObj *cMP3SearchResult::selected=0;


cMP3SearchResult::cMP3SearchResult(const char *szTitle)
:cOsdMenu(szTitle ,GetTab(1), GetTab(2), GetTab(3), GetTab(4)) //, GetTab(5))
{

  if(LoadResult()) {
    SetHelp();
    }
  else
    Skins.Message(mtError, tr("Error while opening @Suchergebnis.m3u !"));

    Display();
}


cMP3SearchResult::~cMP3SearchResult()
{
}


cFileObj *cMP3SearchResult::CurrentItem(void) {
  cMP3SearchBrowseItem *item = (cMP3SearchBrowseItem *)Get(Current());
  return item ? item->Item():0;
}  


void cMP3SearchResult::SetHelp() {
  cOsdMenu::SetHelp(tr("Parent"),tr("Add"),tr("Add all"),tr("ID3info"));
}

int cMP3SearchResult::GetTab(int Tab)
{
  if (Tab==1)    return  5;
  if (Tab==2)    return  30;
  if (Tab==3)    return  8;
  if (Tab==4)    return  7;
  return 0;
}  


int cMP3SearchResult::LoadResult() {

  using namespace std;
  ifstream filestr;
  std::string line;
  std::string SongTitle;
  std::string Base;
  std::string basesource;
  std::string datei;
  
  int count, maxcount;

  bool result = false;

  basesource=BaseSource;
  int len2 = basesource.length()+1;
  count    = 0;
  maxcount = MP3Setup.MaxResult;
  datei = basesource;


  datei = datei + "/@Suchergebnis.m3u";

  Skins.Message(mtStatus, tr("Searching..."));
  
  filestr.open (datei.c_str());

printf("GETLINE GETLINE GETLINE\n");

  if(filestr) {
    while ((getline(filestr,line,'\n')) && (count < maxcount) ) {
      count++;
// Songname

      int len = line.length();
      string::size_type pos = line.rfind ('/',len);
      if(pos!=string::npos) {
        SongTitle =  line.substr(pos+1,len);
// Basedir
        int sublen = len - len2 - SongTitle.length()-1;
        Base   =  line.substr(len2,sublen);
        }


      newsong=new cFileObj(MP3Sources.GetSource(), Base.c_str(), SongTitle.c_str(),otFile);

      if(newsong && newsong->Type()==otFile) {
        Add(new cMP3SearchBrowseItem(newsong, count));
        }
      else
        delete newsong;       
    }

  filestr.close();

printf("STOPGETLINE STOPGETLINE STOPGETLINE\n");
  
  result = true;
  }

  return result;
}


eOSState cMP3SearchResult::ID3Info(void) {
  cFileObj *item=CurrentItem();
  if(item && item->Type()==otFile) {
    cSong *song=new cSong(item);
    cSongInfo *si;
    if(song && (si=song->Info())) {
      AddSubMenu(new cMP3SearchID3Info(song));
      }
    delete song;
  }

  return osContinue;
}        


eOSState cMP3SearchResult::AddOne(void) {
  cFileObj *item=CurrentItem();
  
  if((item=CurrentItem())) {
    selected=new cFileObj(item);
    cInstantPlayList *newpl= new cInstantPlayList(item);
    if(newpl->Load() && newpl->Count()) {
      isyslog("mp3ng: added track %s to playlist %s", item->Name(),newpl->Name());
      cMP3Control::SetPlayList(newpl);

      cMP3SearchBrowseItem *cur = (cMP3SearchBrowseItem *)Get(Current());
      cur->NewName();
      Display();

      return osContinue;
      }
    else
      delete newpl;
    }

  return osContinue;    
}      



eOSState cMP3SearchResult::AddAll(void) {
  Skins.Message(mtStatus, tr("Add all tracks to playlist..."));

  cFileObj *item=new cFileObj(MP3Sources.GetSource(), 0, "@Suchergebnis.m3u",otFile);
  cPlayList *newpl=new cPlayList(item);

  if(newpl->Load() && newpl->Count()) {
    isyslog("mp3ng: added all tracks to playlist %s", newpl->Name());
    cMP3Control::SetPlayList(newpl);
    Skins.Message(mtStatus, NULL);
    return osBack;
    }

  delete newpl;
  Skins.Message(mtStatus, NULL);

  return osContinue;
}      


eOSState cMP3SearchResult::ProcessKey(eKeys Key) {

  bool hadSubmenu = HasSubMenu();

  eOSState state = cOsdMenu::ProcessKey(Key);

  if (hadSubmenu && !HasSubMenu())
    return osContinue;

  if (state == osUnknown) {
    switch (Key) {

      case kRed:
	  return osBack;
	break;

      case kGreen:	  
          state = AddOne();
	break;

      case kYellow:
           state = AddAll();
        break;

      case kBlue:
      case kOk:
           state = ID3Info();
	break;   

      default:
           state = osContinue;
    }
  }
  
  return state;
}        		


// ------------ cMP3SearchValue ------------------------------------------------------------------------------------------------------------

char *cMP3SearchValue::searchtext1 = NULL;
char *cMP3SearchValue::searchtext2 = NULL;
char *cMP3SearchValue::searchtext3 = NULL;
char *cMP3SearchValue::searchtext4 = NULL;


bool equal = false;

cMP3SearchValue::cMP3SearchValue(const char *Title , int which)
:cOsdMenu(tr("Search for..."), 10)
{
  Which=which;
  equal = false;
  SetHelp();
printf("1a\n");
  free(searchtext1); searchtext1=0;
printf("2a\n");
  free(searchtext2); searchtext2=0;
printf("3a\n");
  free(searchtext3); searchtext3=0;
printf("4a\n");
  free(searchtext4); searchtext4=0;
 
printf("5a\n");
  data1[0]=0;
printf("6a\n");
  data2[0]=0;
printf("7a\n");
  data3[0]=0;
printf("8a\n");
  data4[0]=0;
printf("9a\n");

  switch(Which) {
    case 0:  Add(new cMenuEditStrItem(Title, data1 , sizeof(data1)-1, tr(FileNameChars)));
             break;
    case 1:  Add(new cMenuEditStrItem(Title, data1 , sizeof(data1)-1, tr(FileNameChars)));
             break;
    case 2:  Add(new cMenuEditStrItem(Title, data1 , sizeof(data1)-1, tr(FileNameChars)));
             break;
    case 3:  Add(new cMenuEditNumItem(Title, data1 , 320000, false));
             break;
    case 4:  Add(new cMenuEditNumItem(Title, data1 , 2032, false));
             break;
    case 5:  Add(new cMenuEditStrItem(Title, data1 , sizeof(data1)-1, tr(FileNameChars)));
             break;
    case 6:  Add(new cMenuEditStrItem(Title, data1 , sizeof(data1)-1, tr(FileNameChars)));
             SetGenres();
             break;
    // Genre->Artst
    case 7:  Add(new cMenuEditStrItem(Title, data1 , sizeof(data1)-1, tr(FileNameChars)));
             Add(new cMenuEditStrItem(tr("Genre"), data2 , sizeof(data2)-1, tr(FileNameChars)));
             SetGenres();
             break;
    case 8:  Add(new cMenuEditNumItem(Title, data1 , 3600, false));
             break;
    case 9:  Add(new cMenuEditNumItem(Title, data1 , 3600, false));
             break;
    case 10:
printf("10a\n");
             Add(new cMenuEditStrItem("Artist", data1 , sizeof(data1)-1, tr(FileNameChars)));  // Artist
printf("11a\n");
             Add(new cMenuEditStrItem("Genre",  data2 , sizeof(data2)-1, tr(FileNameChars)));  // Genre
printf("12a\n");
/*
             Add(new cMenuEditInttem("Year",   data3 , 2032,false ));                          // Year
printf("13a\n");
             rate[0]=tr("All ratings");
             rate[1]=tr("Not rated");
             rate[2]=tr("Marked songs");
             rate[3]=tr("Bad song not not deleted");
             rate[4]=tr("Only bad songs");
             rate[5]=tr("Not my taste");
             rate[6]=tr("Can hear from time to time");
             rate[7]=tr("New songs");
             rate[8]=tr("Average songs");
             rate[9]=tr("Good songs");
             rate[10]=tr("Very good songs");
             rate[11]=tr("Great songs you can hear often");
             rate[12]=tr("Top songs");
             Add(new cMenuEditStraItem("Rating", data4 , 13, rate));                            // Rating
printf("14a\n");
*/
	     break;
    case 11: Add(new cMenuEditNumItem(Title, data1 , 9, false));
             GetRatingSearchText();
             break;
	     
  }
}  


cOsdItem *cMP3SearchValue::Item(const char *text)
{
  char *buf=0;
  asprintf(&buf,"%s",text?text:"");
  cOsdItem *item = new cOsdItem(buf,osBack);
  item->SetSelectable(false);
  free(buf);
  Add(item); return item;
}



void cMP3SearchValue::SetHelp() {
  if(Which==10)
    cOsdMenu::SetHelp(tr("Parent"),NULL,"Get Genre","Execute");
  else
    cOsdMenu::SetHelp(tr("Parent"),NULL,NULL,NULL);
  
}


void cMP3SearchValue::GetRatingSearchText() {
  std::ifstream filestr;
  std::string datei;
  std::string line;
  int count=0;

  datei=cPlugin::ConfigDirectory(i18n_name);
  datei=datei + "/data/ratingsearchtext.dat";

  filestr.open (datei.c_str());

  
  if(filestr) {
    while ((getline(filestr,line,'\n'))) {
      count++;
      char content[64];
      strcpy(content,line.c_str()); 
      Item(content);
    }
    
    filestr.close();
  }
}



void cMP3SearchValue::SetGenres() {
  std::ifstream filestr;
  std::string datei;
  std::string line;
  int count=0;

  datei=cPlugin::ConfigDirectory(i18n_name);
  datei=datei + "/data/genre.dat";

  filestr.open (datei.c_str());

  
  if(filestr) {
    while ((getline(filestr,line,'\n'))) {
      count++;
      char content[64];
      strcpy(content,line.c_str()); 
      Add(new cOsdItem(hk(content)));
    }
    
    filestr.close();
  }
  else printf("mp3ng: File not found ,%s\n",datei.c_str());

}


void cMP3SearchValue::GetTheValues() {

  int current=Current();


  if(Which==10) {

printf("15a\n");
    if(data1[0]) searchtext1=strdup(data1);
printf("16a\n");
    if(data2[0]) searchtext2=strdup(data2);
printf("17a\n");
/*    switch(data4) {
      case  0: searchtext4=-2;
      case  1: searchtext4=-1;
      case  2: searchtext4=3;
      case  3: searchtext4=28;
      case  4: searchtext4=53;
      case  5: searchtext4=78;
      case  6: searchtext4=104;
      case  7: searchtext4=129;
      case  8: searchtext4=154;
      case  9: searchtext4=179;
      case 10: searchtext4=205;
      case 11: searchtext4=230;
      case 12: searchtext4=255;
      default: break;
    }  
      
printf("18a\n");
    if(data3>1900) searchtext3=data4;
    else searchtext3=-1;
printf("19a\n"); */
  } 
  else {
    if(data1[0]) {
      searchtext1=strdup(data1);
      equal=false;
      }
    else {
      if(Which==6) {    
        cOsdItem *item = Get(current);
        if(current>=1 && (item)){
          searchtext1=strdup(item->Text());
          equal=true;
        }
      }  
    }

  
    if(data2[0]) {
      searchtext2=strdup(data2);
      equal=false;
      }
    else {
      cOsdItem *item = Get(current);
      if(current>=2 && (item) && (Which==7)){
        searchtext2=strdup(item->Text());
        equal=true;
      }  
    }    
  }
}

eOSState cMP3SearchValue::ProcessKey(eKeys Key)
{

  eOSState state = cOsdMenu::ProcessKey(Key);
  
  if(state == osUnknown){
    switch (Key) {

      case kOk:   
          GetTheValues();
          return osBack;
          break;
      case kRed:
      case kBack: state = osBack;
          break;
      default: state = osContinue;		        
      }
  }      
  return state;
}


//---------------- cMP3SearchMoreValue --------------------------------------------------------------------------------------------------------------------------
/*
char *cMP3SearchMoreValue::searchtext1 = NULL;
char *cMP3SearchMoreValue::searchtext2 = NULL;
char *cMP3SearchMoreValue::searchtext3 = NULL;
char *cMP3SearchMoreValue::searchtext4 = NULL;


cMP3SearchMoreValue::cMP3SearchMoreValue(const char *Title)
:cOsdMenu(Title)
{
  equal = false;

  free(searchtext1); searchtext1=0;
  free(searchtext2); searchtext2=0;
  free(searchtext3);searchtext3=0;
  free(searchtext4);searchtext4=0;
 
  data1[0]=0;
  data2[0]=0;
  data3=0;
  data4=0;

  Add(new cMenuEditStrItem("Artist", data1 , sizeof(data1)-1, tr(FileNameChars)));  // Artist
  Add(new cMenuEditStrItem("Genre",  data2 , sizeof(data2)-1, tr(FileNameChars)));  // Genre
  Add(new cMenuEditNumItem("Year",   data3 , 2032, false));                         // Year
  Add(new cMenuEditNumItem("Rating", data4 , 9, false));                            // Rating

  SetHelp();
}
*/
/*
void cMP3SearchMoreValue::SetHelp() {
  cOsdMenu::SetHelp(tr("Parent"),tr("Get Rating"),tr("Get Genre"),tr("Execute"));
}
*/
/*
void cMP3SearchMoreValue::GetTheValues() {

  int current=Current();

  if(data1[0]) searchtext1=strdup(data1);
  if(data2[0]) searchtext2=strdup(data2);
  if(data3>0) searchtest3=itoa(data3);
  if(data4 >0) searchtest4=itoa(data4);
}
*/



/*
eOSState cMP3SearchMoreValue::ProcessKey(eKeys Key)
{

  eOSState state = cOsdMenu::ProcessKey(Key);
  
  if(state == osUnknown){
    switch (Key) {

      case kOk:   
          GetTheValues();
          return osBack;
          break;
      case kGreen: break;
      case kYellow: break;
      case kRed:
      case kBack: state = osBack;
          break;
      default: state = osContinue;		        
      }
  }      
  return state;
}
*/

//---------------- cMP3Search --------------------------------------------------------------------------------------------------------------------------

cMP3Search::cMP3Search(void)
:cOsdMenu(tr("MP3ng Search"))
{
  searching=false;
  SetHelp();

  Add(new cOsdItem(hk(tr("Search for Artist..."))));                   //  0
  Add(new cOsdItem(hk(tr("Search for Title..."))));                    //  1
  Add(new cOsdItem(hk(tr("Search in Filename..."))));                  //  2
  Add(new cOsdItem(hk(tr("Search for Bitrate lower than..."))));       //  3
  Add(new cOsdItem(hk(tr("Search for Year..."))));                     //  4
  Add(new cOsdItem(hk(tr("Search for Album..."))));                    //  5
  Add(new cOsdItem(hk(tr("Search for Genre..."))));                    //  6
  // Genre->Artst
  Add(new cOsdItem(hk(tr("Search in Genre for Artist..."))));          //  7
  Add(new cOsdItem(hk(tr("Search for playlength lower than..."))));    //  8
  Add(new cOsdItem(hk(tr("Search for playlength higher than..."))));   //  9
  Add(new cOsdItem(hk(tr("Search for more then one expressions..."))));// 10 
  Add(new cOsdItem(hk(tr("Search for rated songs"))));                 // 11
  Add(new cOsdItem(hk(tr("Search for marked songs"))));                // 12
  Add(new cOsdItem(hk(tr("Search for unrated songs"))));               // 13
  Add(new cOsdItem(hk(tr("Purge id3Cache instantly"))));               // 14

  Display();
}

cMP3Search::~cMP3Search()
{
}


void cMP3Search::SetHelp() {
  cOsdMenu::SetHelp(tr("Parent"),NULL,NULL,NULL);
}


eOSState cMP3Search::Search(bool second) {

  if(HasSubMenu()) return osContinue;

  int current = Current();

  if(current<=11) {
    if(!second) {
      searching = true;
      std::string szTitle;

      if (current==0) {
        szTitle = tr("Artist");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 0));
	}
      else if (current==1) {	
        szTitle = tr("Title");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 1));
	}
      else if (current==2) {	
        szTitle = tr("Filename");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 2));
	}
      else if (current==3) {	
        szTitle = tr("Bitrate [bps]");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 3));
        }
      else if (current==4) {	
        szTitle = tr("Year");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 4));
	}
      else if (current==5) {	
        szTitle = tr("Album");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 5));
	}
      else if (current==6) {	
        szTitle = tr("Genre");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 6));
	}
      else if (current==7) {	
        szTitle = tr("Artist");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 7));
	}
      else if (current==8) {	
        szTitle = tr("Minutes");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 8));
	}
      else if (current==9) {	
        szTitle = tr("Minutes");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 9));
        }
      else if (current==10) {	
        szTitle = tr("Search for...");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 10));
        }
      else if (current==11) {	
        szTitle = tr("Rating");
        return AddSubMenu(new cMP3SearchValue(szTitle.c_str(), 11));
        }

    }




    searching=false;
printf("1\n");
    const char *searchtext1=cMP3SearchValue::GetNewValue1();
printf("2\n");
    const char *searchtext2=cMP3SearchValue::GetNewValue2();
printf("3\n");
    const char *searchtext3=cMP3SearchValue::GetNewValue3();
printf("4\n");
    const char * searchtext4=cMP3SearchValue::GetNewValue4();
printf("5\n");

    
    if(searchtext1 || searchtext2 || searchtext3 || searchtext4) {
printf("6\n");
      if(InfoCache.Search(current, equal , searchtext1, searchtext2, searchtext3, searchtext4)) {
printf("7\n");
        char *buf;
printf("8\n");
        asprintf(&buf,"%s %i", tr("Found:"), InfoCache.expr1);
printf("9\n");
        strcpy(searchresult,buf);
printf("10\n");
        free(buf);    
printf("11\n");
        return AddSubMenu(new cMP3SearchResult(searchresult));
        }
      else
        return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not find anything !"), fontFix));
    }

  }
  else if(current==12)  {
    searching=false;

    if(InfoCache.Search(current, false , NULL, NULL, NULL, NULL)) {
      char *buf;
      asprintf(&buf,"%s %i", tr("Found:"), InfoCache.expr1);
      strcpy(searchresult,buf);
      free(buf);    
      return AddSubMenu(new cMP3SearchResult(searchresult));
      }
    else
      return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not find anything !"), fontFix));
  }
  else if(current==13) {
    searching=false;

    if(InfoCache.Search(current, false , NULL, NULL, NULL, NULL)) {
      char *buf;
      asprintf(&buf,"%s %i", tr("Found:"), InfoCache.expr1);
      strcpy(searchresult,buf);
      free(buf);    
      return AddSubMenu(new cMP3SearchResult(searchresult));
      }
    else
      return AddSubMenu(new cMenuText(tr("Error:"), tr("ERROR: Could not find anything !"), fontFix));
  
  }
  else {
    searching=false;

    if(current==14) {
        Skins.Message(mtStatus, tr("Purge id3Cache..."));
        InfoCache.PurgeNow();
        Skins.Message(mtStatus, NULL);
	return AddSubMenu(new cMenuText(tr("Purge id3Cache"), tr("id3Cache purged !"), fontSml));
    }
  }    

  return osContinue;
}	 
    

eOSState cMP3Search::ProcessKey(eKeys Key)
{
  eOSState state = cOsdMenu::ProcessKey(Key);
  
  if(!HasSubMenu() && state==osContinue) {
    if(searching) return Search(true);
  }

  if (state == osUnknown) {
    switch (Key) {
        case kBack:
        case kRed: state = osBack; 
          break;
	case kGreen: 
	case kYellow: 
	case kBlue: return osContinue;
	  break;
        case kOk:   state = Search(false);
	  break;
        default: state = osContinue;
    }
  }
  return state;
}


