/*
 * Softcam plugin to VDR (C++)
 *
 * This code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <libsi/section.h>

#include "viaccess.h"
#include "tps.h"
#include "sc.h"
#include "scsetup.h"
#include "system-common.h"
#include "filter.h"
#include "misc.h"
#include "log-viaccess.h"

#define CHECK_TIME    5*60*1000
#define LOADBIN_TIME 60*60*1000
#define TPSAU_TIME   30*60*1000

//#define TRUST_BOXTIME // should we trust the local time on this machine?
//#define DUMP_TPSAU "/var/tmp"

// -- cRC6 ---------------------------------------------------------------------

/*
 * This code implements the RC6-32/20 block cipher.
 *
 * The algorithm is due to Ron Rivest and RSA Labs. This code is based on code
 * which was written by Martin Hinner <mhi@penguin.cz> in 1999, no copyright is
 * claimed.
 */

#define RC6_WORDSIZE	32
#define RC6_P32		0xB7E15163L
#define RC6_Q32		0x9E3779B9L

unsigned int cRC6::rol(unsigned int v, unsigned int cnt)
{
  cnt&=(RC6_WORDSIZE-1);
  return (v<<cnt) | (v>>(RC6_WORDSIZE-cnt));
}

unsigned int cRC6::ror(unsigned int v, unsigned int cnt)
{
  cnt&=(RC6_WORDSIZE-1);
  return (v>>cnt) | (v<<(RC6_WORDSIZE-cnt));
}

void cRC6::SetKey(const unsigned char *Key, int len)
{
  key[0]=RC6_P32;
  for(int v=1; v<RC6_MAX; v++) key[v]=key[v-1]+RC6_Q32;
  len/=4;
  unsigned int a=0, b=0, l[len];
  memcpy(l,Key,len*4);
  for(int i=0,j=0,v=3*(len>RC6_MAX ? len : RC6_MAX) ; v>0; v--) {
    a=key[i]=rol(key[i]+a+b,3);
    b=  l[j]=rol(  l[j]+a+b,a+b);
    i++; i%=RC6_MAX;
    j++; j%=len;
    }
}

void cRC6::Decrypt(unsigned char *data)
{
  unsigned int *l=(unsigned int *)data;
  unsigned int a, b, c, d;
  a=l[0]-key[RC6_MAX-2];
  b=l[1];
  c=l[2]-key[RC6_MAX-1];
  d=l[3];
  for(int i=RC6_ROUNDS; i>0; i--) {
    unsigned int t=a;
    unsigned int u=b;
    a=d; d=c; b=t; c=u;
    u=rol((d*(2*d+1)),5);
    t=rol((b*(2*b+1)),5);
    c=ror(c-key[2*i+1],t)^u;
    a=ror(a-key[2*i  ],u)^t;
    }
  l[0]=a;
  l[1]=b-key[0];
  l[2]=c;
  l[3]=d-key[1];
}

// -- cTransponderTime ---------------------------------------------------------

class cTransponderTime : public cSimpleItem {
private:
  time_t sattime;
  cTimeMs reftime;
  cMutex mutex;
  cCondVar wait;
  int source, transponder;
  bool hasHandler;
public:
  cTransponderTime(int Source, int Transponder);
  bool Is(int Source, int Transponder) const;
  void Set(time_t st);
  time_t Now(void);
  void SetHasHandler(bool on) { hasHandler=on; }
  bool HasHandler(void) const { return hasHandler; }
  };

cTransponderTime::cTransponderTime(int Source, int Transponder)
{
  source=Source; transponder=Transponder;
#ifdef TRUST_BOXTIME
  sattime=time(0);
#else
  sattime=-1;
#endif
  hasHandler=false;
}

bool cTransponderTime::Is(int Source, int Transponder) const
{
  return source==Source && transponder==Transponder;
}

void cTransponderTime::Set(time_t st)
{
  cMutexLock lock(&mutex);
  if(sattime<0 || reftime.Elapsed()>5*1000) {
    reftime.Set(-100);
    sattime=st;
    wait.Broadcast();

    char str[32];
    struct tm utm;
    st=Now();
    time_t now=time(0);
    gmtime_r(&st,&utm); asctime_r(&utm,str); stripspace(str);
    PRINTF(L_SYS_TIME,"%x:%x: %s (delta %ld)",source,transponder,str,st-now);
    }
}

time_t cTransponderTime::Now(void)
{
  cMutexLock lock(&mutex);
  if(sattime<0) {
    if(!wait.TimedWait(mutex,3*1000)) {
      PRINTF(L_SYS_TIME,"%x:%x: unsuccessfull wait",source,transponder);
      return -1;
      }
    }
  return sattime+(reftime.Elapsed()/1000);
}

// -- cSatTimeHook -------------------------------------------------------------

class cSatTimeHook : public cLogHook {
private:
  cTransponderTime *ttime;
public:
  cSatTimeHook(cTransponderTime *Ttime);
  ~cSatTimeHook();
  virtual void Process(int pid, unsigned char *data);
  };

cSatTimeHook::cSatTimeHook(cTransponderTime *Ttime)
:cLogHook(HOOK_SATTIME,"sattime")
{
  ttime=Ttime;
  pids.AddPid(0x14,0x71,0xff,0x03);
  ttime->SetHasHandler(true);
}

cSatTimeHook::~cSatTimeHook()
{
  ttime->SetHasHandler(false);
}

void cSatTimeHook::Process(int pid, unsigned char *data)
{
  if(data && ttime) {
    if(data[0]==0x70) { // TDT
      SI::TDT tdt(data,false);
      tdt.CheckParse();
      ttime->Set(tdt.getTime());
      }
    else if(data[0]==0x73) { // TOT
      SI::TOT tot(data,false);
      if(!tot.CheckCRCAndParse()) return;
      ttime->Set(tot.getTime());
      }
    }
}

// -- cSatTime -----------------------------------------------------------------

class cSatTime {
private:
  static cMutex mutex;
  static cSimpleList<cTransponderTime> list;
  //
  int cardNum;
  cTransponderTime *ttime;
  //
  void CheckHandler(void);
public:
  cSatTime(int CardNum, int Source, int Transponder);
  time_t Now(void);
  };

cMutex cSatTime::mutex;
cSimpleList<cTransponderTime> cSatTime::list;

cSatTime::cSatTime(int CardNum, int Source, int Transponder)
{
  cardNum=CardNum;
  cMutexLock lock(&mutex);
  for(ttime=list.First(); ttime; ttime=list.Next(ttime))
    if(ttime->Is(Source,Transponder)) break;
  if(!ttime) {
    ttime=new cTransponderTime(Source,Transponder);
    if(ttime) list.Add(ttime);
    PRINTF(L_SYS_TIME,"%x:%x: created new transpondertime",Source,Transponder);
    }
  else PRINTF(L_SYS_TIME,"%x:%x: using existing transpondertime",Source,Transponder);
  CheckHandler();
}

time_t cSatTime::Now(void)
{
  CheckHandler();
  return ttime ? ttime->Now() : -1;
}

void cSatTime::CheckHandler(void)
{
  if(ttime) {
    if(!cSoftCAM::TriggerHook(cardNum,HOOK_SATTIME) && !ttime->HasHandler()) {
      cSatTimeHook *hook=new cSatTimeHook(ttime);
      cSoftCAM::AddHook(cardNum,hook);
      PRINTF(L_SYS_TIME,"added hook");
      }
    }
}

// -- cTpsKey ------------------------------------------------------------------

cTpsKey::cTpsKey(void)
{
  timestamp=0;
}

void cTpsKey::Set(const cTpsKey *k)
{
  timestamp=k->timestamp;
  opmode=k->opmode;
  memcpy(step,k->step,sizeof(step));
}

void cTpsKey::Set(const unsigned char *mem)
{
  timestamp=*((unsigned int *)mem);
  opmode=mem[52+3];
  for(int i=0; i<3; i++) {
    memcpy(step[i].key,&mem[4+i*16],16);
    step[i].mode=mem[52+i];
    }
}

void cTpsKey::Put(unsigned char *mem) const
{
  *((unsigned int *)mem)=timestamp;
  mem[52+3]=opmode;
  for(int i=0; i<3; i++) {
    memcpy(&mem[4+i*16],step[i].key,16);
    mem[52+i]=step[i].mode;
    }
}

// -- cTpsAuHook ---------------------------------------------------------------

#define BUFF_SIZE 20000

class cTpsAuHook : public cLogHook {
private:
  unsigned char *mem;
  int pos;
  int pmtpid, aupid;
public:
  cTpsAuHook(void);
  ~cTpsAuHook();
  virtual void Process(int pid, unsigned char *data);
  };

// -- cTpsKeys -----------------------------------------------------------------

cTpsKeys tpskeys;

cTpsKeys::cTpsKeys(void)
:cLoader("TpsAu")
,lastLoad(-LOADBIN_TIME)
,lastAu(-TPSAU_TIME)
{
  list=new cSimpleList<cTpsKey>;
  first=last=0;
}

cTpsKeys::~cTpsKeys()
{
  delete list;
}

const cTpsKey *cTpsKeys::GetKey(time_t t)
{
  cMutexLock lock(this);
  for(cTpsKey *k=list->First(); k; k=list->Next(k))
    if(t<k->Timestamp()) return k;
  return 0;
}

void cTpsKeys::Check(time_t now, int cardnum)
{
  checkMutex.Lock();
  if(first==0 && last==0 && list->Count()>0)
    GetFirstLast();
  if(now>0 && lastCheck.Elapsed()>CHECK_TIME) {
    Purge(now);
    lastCheck.Set();
    }
  bool nokey=now+2*3600>last;
  if(lastLoad.Elapsed()>(nokey ? LOADBIN_TIME/60 : LOADBIN_TIME)) {
    PRINTF(L_SYS_TPSAU,"loading "TPSBIN" triggered");
    LoadBin();
    if(now>0) Purge(now);
    lastLoad.Set();
    }
  if(lastAu.Elapsed()>(nokey ? TPSAU_TIME/60 : TPSAU_TIME)) {
    if(ScSetup.AutoUpdate>0) {
      PRINTF(L_SYS_TPSAU,"TPS AU triggered");
      if(!cSoftCAM::TriggerHook(cardnum,HOOK_TPSAU)) {
        cTpsAuHook *hook=new cTpsAuHook;
        cSoftCAM::AddHook(cardnum,hook);
        PRINTF(L_SYS_TPSAU,"TPS AU hook added");
        }
      }
    lastAu.Set();
    }
  checkMutex.Unlock();
}

void cTpsKeys::Purge(time_t now)
{
  cMutexLock lock(this);
  PRINTF(L_SYS_TPSAU,"purging TPS keylist");
  bool del=false;
  for(cTpsKey *k=list->First(); k;) {
    cTpsKey *n=list->Next(k);
    if(k->Timestamp()<now-3600) { list->Del(k); del=true; }
    k=n;
    }
  if(del) { 
    GetFirstLast();
    Modified();
    cLoaders::SaveCache();
    }
}

void cTpsKeys::Join(cSimpleList<cTpsKey> *nlist)
{
  cMutexLock lock(this);
  cTpsKey *k;
  while((k=nlist->First())) {
    nlist->Del(k,false);
    cTpsKey *p=list->First();
    do {
      if(!p) {
        list->Add(k);
        Modified();
        break;
        }
      cTpsKey *n=list->Next(p);
      if(k->Timestamp()==p->Timestamp()) {
        p->Set(k);
        Modified();
        delete k;
        break;
        }
      if(k->Timestamp()>p->Timestamp() && (!n || k->Timestamp()<n->Timestamp())) {
        list->Add(k,p);
        Modified();
        break;
        }
      p=n;
      } while(p);
    }
  delete nlist;
  GetFirstLast();
  cLoaders::SaveCache();
}

cString cTpsKeys::Time(time_t t)
{
  char str[32];
  struct tm tm_r;
  strftime(str,sizeof(str),"%b %e %T",localtime_r(&t,&tm_r));
  return str;
}

void cTpsKeys::GetFirstLast(void)
{
  if(list->Count()>0) {
    cTpsKey *k=list->First();
    first=last=k->Timestamp();
    for(; k; k=list->Next(k)) {
      if(k->Timestamp()<last)
        PRINTF(L_SYS_TPSAU,"TPS keys not in accending order!");
      last=k->Timestamp();
      }
    PRINTF(L_SYS_TPS,"%d TPS keys available (from %s to %s)",list->Count(),*Time(first),*Time(last));
    }
  else {
    last=first=0;
    PRINTF(L_SYS_TPS,"no TPS keys available");
    }
}

bool cTpsKeys::ProcessAu(unsigned char *mem, int len)
{
  PRINTF(L_SYS_TPSAU,"processing TPS AU data (len=%d)",len);
#ifdef DUMP_TPSAU
#warning Dumping TPS AU data
  {
  char fname[32];
  snprintf(fname,sizeof(fname),"%s/tpsau.dump.%d.%x",DUMP_TPSAU,getpid(),(int)time(0));
  int fd=open(fname,O_CREAT|O_TRUNC|O_WRONLY);
  if(fd>=0) {
    write(fd,mem,len);
    close(fd);
    PRINTF(L_SYS_TPSAU,"dumped to file '%s'",fname);
    }
  }
#endif
  unsigned char *data;
  int seclen, numkeys;
  for(int i=0; ; i++) {
    static const unsigned char startcode[] = {
      0x6F,0x61,0x64,0x63,0x61,0x73,0x74,0x2E,0x63,0x68,0x61,0x6E,0x6E,0x65,0x6C,0x2E,
      0x31,0x33,0x35,0x2E,0x34,0x35,0x31,0x09,0x05,0x4F,0x00,0x00,0x00,0x00,0x00 };
    if(memcmp(&mem[i],startcode,sizeof(startcode))==0) {
      data=&mem[i+32];
      seclen=data[0] | (data[1]<<8);
      numkeys=data[2] | (data[3]<<8);
      if(seclen>4 && numkeys>0) break;
      }
    if(i>len-200) {
      PRINTF(L_SYS_TPSAU,"no startcode found");
      return false;
      }
    }
  int algo=data[4];
  int mkidx=data[5]&7;
  unsigned char *sec[7];
  sec[0]=data+6;
  for(int i=1; i<6; i++) sec[i]=sec[i-1]+seclen;
  sec[6]=sec[5]+numkeys;
  PRINTF(L_SYS_TPSAU,"found startcode (off=%d seclen=%d numkeys=%d keyoff=%d algo=%d mkidx=%d)",data-mem,seclen,numkeys,sec[6]-mem,algo,mkidx);
  unsigned char key[16];
  cPlainKey *pk=keys.FindKey('V',0x7c00,MBC3('M','K',mkidx),16);
  if(!pk) {
    PRINTF(L_SYS_KEY,"missing V 7C00 TPSMK%d key",mkidx);
    return false;
    }
  pk->Get(key);
  if(sec[6]>=mem+len) {
    PRINTF(L_SYS_TPSAU,"section 5 exceeds buffer");
    return false;
    }
  int keylen=0;
  for(int i=0; i<numkeys; i++) keylen+=sec[5][i];
  keylen=(keylen+15)&~15;
  if(sec[6]+keylen>=mem+len) {
    PRINTF(L_SYS_TPSAU,"section 6 exceeds buffer");
    return false;
    }
  for(int i=0; i<keylen; i+=16) TpsDecrypt(&sec[6][i],algo,key);

  cSimpleList<cTpsKey> *nlist=new cSimpleList<cTpsKey>;
  for(int i=0; i<seclen; i++) {
    static const unsigned char startkey[] = { 0x01,0x01,0x03 };
    static const unsigned char startaes[] = { 0x09,0x10 };
    static const unsigned char startse[] = { 0x0a,0x10 };
    unsigned char tmp[56];
    tmp[0]=sec[0][i];
    tmp[1]=sec[1][i];
    tmp[2]=sec[2][i];
    tmp[3]=sec[3][i];
    if(CheckFF(tmp,4)) continue;
    int keyid=sec[4][i];
    int keylen=sec[5][keyid];
    if(keylen<45) continue;
    unsigned char *tkey=sec[6];
    for(int j=0; j<keyid; j++) tkey+=sec[5][j];
    if(memcmp(tkey,startkey,sizeof(startkey))) continue;
    tmp[52]=0;
    tmp[53]=tkey[4];
    tmp[54]=tkey[5];
    tmp[55]=0x1c;
    tkey+=9;
    if(memcmp(tkey,startaes,sizeof(startaes))) continue;
    memset(&tmp[4+ 0],0,16);
    memcpy(&tmp[4+16],&tkey[2],16);
    tkey+=18;
    if(memcmp(tkey,startse,sizeof(startse))) continue;
    memcpy(&tmp[4+32],&tkey[2],16);
    cTpsKey *k=new cTpsKey;
    if(k) { k->Set(tmp); nlist->Add(k); }
    }
  PRINTF(L_SYS_TPSAU,"got %d keys from AU data",nlist->Count());
  bool res=nlist->Count()>0;
  Join(nlist);
  return res;
}

bool cTpsKeys::LoadBin(void)
{
  static const unsigned char mark[] = { 'T','P','S',0 };
  cFileMap *tpsbin=filemaps.GetFileMap(TPSBIN,FILEMAP_DOMAIN,false);
  if(!tpsbin) {
    PRINTF(L_SYS_TPS,"no filemap for "TPSBIN);
    return false;
    }
  else if(!tpsbin->Map()) {
    PRINTF(L_SYS_TPS,"mapping failed for "TPSBIN);
    return false;
    }
  else if(tpsbin->Size()<65536 || memcmp(mark,tpsbin->Addr()+32,sizeof(mark)) || memcmp(mark,tpsbin->Addr()+48,sizeof(mark))) {
    PRINTF(L_SYS_TPS,TPSBIN" format not recognised");
    tpsbin->Unmap();
    return false;
    }

  int size=tpsbin->Size()-56;
  cSimpleList<cTpsKey> *nlist=new cSimpleList<cTpsKey>;
  for(int i=68; i<size; i+=56) {
    const unsigned char *a=tpsbin->Addr()+i;
    if(*((const unsigned int *)a)==0x00000000L || *((const unsigned int *)a)==0xFFFFFFFFL)
      break;
    unsigned char tmp[56];
    DecryptBin(a,tmp);
    cTpsKey *k=new cTpsKey;
    if(k) { k->Set(tmp); nlist->Add(k); }
    }
  tpsbin->Unmap();
  PRINTF(L_SYS_TPSAU,"loaded %d keys from "TPSBIN" file",nlist->Count());
  Join(nlist);
  return true;
}

void cTpsKeys::DecryptBin(const unsigned char *in, unsigned char *out)
{
  unsigned int var2=*((unsigned int *)in);
  *((unsigned int *)out)=var2;
  for(int i=0; i<13; i++) {
    in+=4; out+=4;
    var2=(var2<<3) | (var2>>(32-3));
    unsigned int var1=*((unsigned int *)in) ^ var2;
    *((unsigned int *)out)=(var1<<(i+2)) | (var1>>(32-(i+2)));
    }
}

bool cTpsKeys::ParseLine(const char *line, bool fromCache)
{
  unsigned char tmp[60];
  if(GetHex(line,tmp,sizeof(tmp))) {
    unsigned int crc=crc32_le(0,&tmp[4],sizeof(tmp)-4);
    if(*((unsigned int *)tmp)==crc) {
      cTpsKey *k=new cTpsKey;
      if(k) { k->Set(&tmp[4]); list->Add(k); }
      return true;
      }
    else PRINTF(L_SYS_TPS,"CRC failed during cache load");
    }
  return false;
}

bool cTpsKeys::Save(FILE *f)
{
  cMutexLock lock(this);
  bool res=true;
  for(cTpsKey *k=list->First(); k; k=list->Next(k)) {
    unsigned char tmp[60];
    k->Put(&tmp[4]);
    *((unsigned int *)tmp)=crc32_le(0,&tmp[4],sizeof(tmp)-4);
    char str[128];
    fprintf(f,"%s\n",HexStr(str,tmp,sizeof(tmp)));
    res=(ferror(f)==0 && res);
    }
  Modified(!res);
  return res;
}

// -- cTpsAuHook ---------------------------------------------------------------

#define AUSID 0x12C0

cTpsAuHook::cTpsAuHook(void)
:cLogHook(HOOK_TPSAU,"tpsau")
{
  mem=MALLOC(unsigned char,BUFF_SIZE);
  pos=0; pmtpid=aupid=-1;
  pids.AddPid(0x0000,0x00,0xff); // PAT
/*
  aupid=0x12f2;
  pids.AddPid(aupid,0x87,0xff);
*/
}

cTpsAuHook::~cTpsAuHook()
{
  free(mem);
}

void cTpsAuHook::Process(int pid, unsigned char *data)
{
  if(data && mem) {
    if(pid==0) { // PAT
      SI::PAT pat(data,false);
      if(pat.CheckCRCAndParse()) {
        SI::PAT::Association assoc;
        for(SI::Loop::Iterator it; pat.associationLoop.getNext(assoc,it);) {
          if(!assoc.isNITPid() && assoc.getServiceId()==AUSID) {
            pmtpid=assoc.getPid();
            PRINTF(L_SYS_TPSAU,"got PMT pid %04x for SID %04x",pmtpid,AUSID);
            cPid *pid=pids.First();
            if(pid && pid->filter) {
              pid->filter->Start(pmtpid,0x02,0xFF,0x00,false);
              pid->filter->Flush();
              }
            else PRINTF(L_GEN_DEBUG,"internal: no pid/filter in cTpsAuHook/pat");
            return;
            }
          }
        PRINTF(L_SYS_TPSAU,"no PMT pid found for SID %04x",AUSID);
        BailOut();
        }
      }
    else if(pid==pmtpid) { // PMT
      SI::PMT pmt(data,false);
      if(pmt.CheckCRCAndParse() && pmt.getServiceId()==AUSID) {
        SI::PMT::Stream stream;
        for(SI::Loop::Iterator it; pmt.streamLoop.getNext(stream,it); ) {
          if(stream.getStreamType()==0x05) {
            aupid=stream.getPid();
            PRINTF(L_SYS_TPSAU,"got AU pid %04x",aupid);
            cPid *pid=pids.First();
            if(pid && pid->filter) {
              pid->filter->Start(aupid,0x87,0xFF,0x00,false);
              pid->filter->Flush();
              }
            else PRINTF(L_GEN_DEBUG,"internal: no pid/filter in cTpsAuHook/pmt");
            return;
            }
          }
        PRINTF(L_SYS_TPSAU,"could not locate AU pid in PMT %04x data",pmtpid);
        BailOut();
        }
      }
    else if(pid==aupid) {
      int len=SCT_LEN(data)-28;
      if(len>0 && pos+len<BUFF_SIZE) {
        memcpy(mem+pos,data+24,len);
        pos+=len;
        }
      if(pos>(BUFF_SIZE-BUFF_SIZE/10)) {
        if(tpskeys.ProcessAu(mem,pos)) BailOut();
        pos-=BUFF_SIZE/2;
        memmove(mem,mem+BUFF_SIZE/2,pos);
        }
      }
    }
}

// -- cTPSDecrypt --------------------------------------------------------------

void cTPSDecrypt::TpsDecrypt(unsigned char *data, short mode, const unsigned char *key)
{
  switch(mode) {
    case 0:
    case 3: break;
    case 1: cAES::SetKey(key); cAES::Decrypt(data,16); break;
    case 2: cRC6::SetKey(key,16); cRC6::Decrypt(data); break;
    default: PRINTF(L_SYS_TPS,"unknown TPS decryption algo %d",mode); break;
    }
}

// -- cTPS ---------------------------------------------------------------------

cTPS::cTPS(void)
{
  sattime=0;
}

cTPS::~cTPS()
{
  delete sattime;
}

int cTPS::Decrypt(int cardNum, int Source, int Transponder, unsigned char *data, int len)
{
  if(!sattime) {
    sattime=new cSatTime(cardNum,Source,Transponder);
    if(!sattime) {
      PRINTF(L_SYS_TPS,"failed to create time class");
      return -1;
      }
    }
  time_t now=sattime->Now();
  tpskeys.Check(now,cardNum);
  if(now<0) return -1;

  const cTpsKey *k=tpskeys.GetKey(now);
  if(!k) {
    PRINTF(L_SYS_TPS,"no TPS key for current time. Outdated "TPSBIN" file?");
    return -1;
    }
  doPost=0;
  int opmode=k->Opmode(), doTPS=0, doPre=0, hasDF=0, ret=0;
  if((opmode&4) && data[0]==0xD2 && data[1]==0x01 && data[2]==0x01) {
    data+=3; len-=3; ret=3;
    if(data[0]==0x90) PRINTF(L_SYS_TPS,"TPS v1 is no longer supported");
    if(data[0]==0x40) data[0]=0x90;
    doTPS=1;
    }
  for(int i=0; i<len; i+=data[i+1]+2)
    switch(data[i]) {
      case 0xDF:
        if(!(opmode&4)) doTPS =(0x6996>>((data[i+2]&0xF)^(data[i+2]>>4)))&1;
        if(opmode&8)    doPre =(0x6996>>((data[i+3]&0xF)^(data[i+3]>>4)))&1;
        if(opmode&16)   doPost=(0x6996>>((data[i+4]&0xF)^(data[i+4]>>4)))&1;
        hasDF=1;
        break;
      case 0xEA:
        if(doPre) TpsDecrypt(&data[i+2],k->Mode(0),k->Key(0));
        if(doTPS) TpsDecrypt(&data[i+2],(hasDF)?k->Mode(1):1,k->Key(1));
        break;
      }
  postMode=k->Mode(2); memcpy(postKey,k->Key(2),sizeof(postKey));
  return ret;
}

void cTPS::PostProc(unsigned char *cw)
{
  if(doPost) TpsDecrypt(cw,postMode,postKey);
}
