/*
 * Softcam plugin to VDR (C++)
 *
 * This code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 */

#ifndef ___CAM_H
#define ___CAM_H

#include <linux/dvb/ca.h>
#include <vdr/dvbdevice.h>
#include <vdr/thread.h>
#include "data.h"
#include "misc.h"

class cChannel;

class cEcmHandler;
class cEcmData;
class cLogger;
class cHookManager;
class cLogHook;
class cDeCSA;
class cDeCsaTSBuffer;
class cScCiAdapter;
class cScDvbDevice;
class cPrg;

// ----------------------------------------------------------------

class cEcmCache : public cLoader, cMutex, cSimpleList<cEcmData> {
private:
  cEcmData *Exists(cEcmInfo *e);
public:
  cEcmCache(void);
  void Load(void);
  void New(cEcmInfo *e);
  int GetCached(cSimpleList<cEcmInfo> *list, int sid, int Source, int Transponder);
  void Delete(cEcmInfo *e);
  void Flush(void);
  virtual bool Save(FILE *f);
  virtual bool ParseLine(const char *line, bool fromCache);
  };

extern cEcmCache ecmcache;

// ----------------------------------------------------------------

#define MAX_CW_IDX        16
#define MAX_CI_SLOTS      8
#define MAX_CI_SLOT_CAIDS 16

class cCam : private cMutex {
private:
  int cardNum;
  cScDvbDevice *device;
  cSimpleList<cEcmHandler> handlerList;
  cLogger *logger;
  cHookManager *hookman;
  int source, transponder, liveVpid, liveApid;
  unsigned char indexMap[MAX_CW_IDX], lastCW[MAX_CW_IDX][2*8];
  //
  cEcmHandler *GetHandler(int sid, bool needZero, bool noshift);
  void RemHandler(cEcmHandler *handler);
  int GetFreeIndex(void);
public:
  cCam(cScDvbDevice *dev, int CardNum);
  virtual ~cCam();
  // EcmHandler API
  void WriteCW(int index, unsigned char *cw, bool force);
  void SetCWIndex(int pid, int index);
  void DumpAV7110(void);
  void LogEcmStatus(const cEcmInfo *ecm, bool on);
  bool GetPrgCaids(int prg, int *c);
  void AddHook(cLogHook *hook);
  bool TriggerHook(int id);
  // Plugin API
  bool Active(void);
  void HouseKeeping(void);
  void Tune(const cChannel *channel);
  void SetPid(int type, int pid, bool on);
  void Stop(void);
  void AddPrg(cPrg *prg);
  char *CurrentKeyStr(int num);
  //
  bool IsSoftCSA(void);
  int Source(void) { return source; }
  int Transponder(void) { return transponder; }
  };

void LogStatsDown(void);

// ----------------------------------------------------------------

#define MAX_LRU_CAID 10

class cScDvbDevice : public cDvbDevice {
private:
  cDeCSA *decsa;
  cDeCsaTSBuffer *tsBuffer;
  cScCiAdapter *ciadapter;
  cCiAdapter *hwciadapter;
  cCam *cam;
  int fd_dvr, fd_ca, fd_ca2;
  bool softcsa;
  cMutex cafdMutex;
  cTimeMs lastDump;
  struct LruCaid {
    int prg, caids[MAXCAIDS+1];
    } lrucaid[MAX_LRU_CAID];
  cMutex lruMutex;
  static int budget;
  //
  void LateInit(void);
  void EarlyShutdown(void);
  int FindLRUPrg(int prg);
protected:
  virtual bool Ready(void);
  virtual bool SetPid(cPidHandle *Handle, int Type, bool On);
  virtual bool SetChannelDevice(const cChannel *Channel, bool LiveView);
  virtual bool OpenDvr(void);
  virtual void CloseDvr(void);
  virtual bool GetTSPacket(uchar *&Data);
public:
  cScDvbDevice(int n, int cafd);
  ~cScDvbDevice();
#if APIVERSNUM >= 10501
  virtual bool HasCi(void);
#endif
  static void Capture(void);
  static bool Initialize(void);
  static void Startup(void);
  static void Shutdown(void);
  static void SetForceBudget(int n);
  static bool ForceBudget(int n);
  bool SetCaDescr(ca_descr_t *ca_descr);
  bool SetCaPid(ca_pid_t *ca_pid);
  void DumpAV7110(void);
  cCam *Cam(void) { return cam; }
  bool SoftCSA(void) { return softcsa; }
  bool GetPrgCaids(int prg, int *c);
  };

#endif // ___CAM_H
