/*
 * mp3ng.c: A plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 */

#include <string>
#include <stdlib.h>
#include <getopt.h>
#include <strings.h>
#include <typeinfo>

#include "mp3ng.h"
#include "common.h"
#include "setup.h"
#include "setup-mp3.h"
#include "data-mp3.h"
#include "player-mp3.h"
#include "decoder-mp3.h"
#include "skin.h"
//#include "bitmap.h"
#include "i18n.h"

cPluginMp3::cPluginMp3(void)
{
  // Initialize any member varaiables here.
  // DON'T DO ANYTHING ELSE THAT MAY HAVE SIDE EFFECTS, REQUIRE GLOBAL
  // VDR OBJECTS TO EXIST OR PRODUCE ANY OUTPUT!
}

cPluginMp3::~cPluginMp3()
{
//  MP3Skin.StoreSkin();
//  cMP3Bitmap::FlushCache();
  InfoCache.Save();
  delete mgr;
}

const char *cPluginMp3::CommandLineHelp(void)
{
  static char *help_str=0;

  free(help_str);    //                                     for easier orientation, this is column 80|
  asprintf(&help_str,"  -m CMD,   --mount=CMD    use CMD to mount/unmount/eject mp3 sources\n"
                     "                           (default: %s)\n"
                     "  -n CMD,   --network=CMD  execute CMD before & after network access\n"
                     "                           (default: %s)\n"
                     "  -C DIR,   --cache=DIR    store ID3 cache file in DIR\n"
                     "                           (default: video dir)\n"
                     "  -B DIR,   --cddb=DIR     search CDDB files in DIR\n"
                     "                           (default: %s)\n"
                     "  -D DIR,   --dsp=DIR      device for OSS output\n"
                     "                           (default: %s)\n",
                     mountscript,netscript?netscript:"none",
#ifdef HAVE_SNDFILE
                     cddbpath,
#else
                     "none",
#endif
#ifdef WITH_OSS
                     dspdevice
#else
                     "none"
#endif
                     );
  return help_str;
}

bool cPluginMp3::ProcessArgs(int argc, char *argv[])
{
  static struct option long_options[] = {
      { "mount",    required_argument, NULL, 'm' },
      { "network",  required_argument, NULL, 'n' },
      { "cddb",     required_argument, NULL, 'B' },
      { "dsp",      required_argument, NULL, 'D' },
      { "cache",    required_argument, NULL, 'C' },
      { NULL }
    };

  int c, option_index = 0;
  while((c=getopt_long(argc,argv,"m:n:B:C:D:",long_options,&option_index))!=-1) {
    switch (c) {
      case 'm': mountscript=optarg; break;
      case 'n': netscript=optarg; break;
      case 'C': cachedir=optarg; break;
      case 'B':
#ifdef HAVE_SNDFILE
                cddbpath=optarg; break;
#else
                fprintf(stderr, "mp3ng: libsndfile support has not been compiled in!\n"); return false;
#endif
      case 'D':
#ifdef WITH_OSS
                dspdevice=optarg; break;
#else
                fprintf(stderr, "mp3ng: OSS output has not been compiled in!\n"); return false;
#endif
      default:  return false;
      }
    }
  return true;
}

bool cPluginMp3::Initialize(void)
{

  if(!CheckVDRVersion(1,1,29,"mp3ng")) return false;

  if(!MP3Skin.ParseSkin("current.colors", false)) {
     esyslog("WARNING [mp3ng]: No themefile found ! Using default colors");
     fprintf(stderr,"mp3ng: no themefile found !\n");
     }
  else isyslog("mp3ng: Loading themefile\n");     
  i18n_name=Name();
  MP3Sources.Load(AddDirectory(ConfigDirectory(),"mp3ng/mp3sources.conf"));
  if(MP3Sources.Count()<1) {
     esyslog("ERROR: you should have defined at least one source in mp3sources.conf");
     fprintf(stderr,"No source(s) defined in mp3sources.conf\n");
     return false;
     }
  InfoCache.Load();
  RegisterI18n(Phrases);
  mgr=new cPlayManager;
  if(!mgr) {
    esyslog("ERROR: creating playmanager failed");
    fprintf(stderr,"Creating playmanager failed\n");
    return false;
    }
  d(printf("mp3ng: using %s\n",mad_version))
  d(printf("mp3ng: compiled with %s\n",MAD_VERSION))

  return true;


}

void cPluginMp3::Housekeeping(void)
{
  InfoCache.Save();
}

const char *cPluginMp3::MainMenuEntry(void)
{
  return MP3Setup.HideMainMenu ? 0 : tr(MAINMENUENTRY);
}

cOsdObject *cPluginMp3::MainMenuAction(void)
{
    return new cMenuMP3;
}

cMenuSetupPage *cPluginMp3::SetupMenu(void)
{
  return new cMenuSetupMP3;
}

bool cPluginMp3::SetupParse(const char *Name, const char *Value)
{
  if      (!strcasecmp(Name, "InitLoopMode"))     MP3Setup.InitLoopMode    = atoi(Value);
  else if (!strcasecmp(Name, "InitShuffleMode"))  MP3Setup.InitShuffleMode = atoi(Value);
  else if (!strcasecmp(Name, "AudioMode"))        MP3Setup.AudioMode       = atoi(Value);
  else if (!strcasecmp(Name, "BgrScan"))          MP3Setup.BgrScan         = atoi(Value);
  else if (!strcasecmp(Name, "EditorMode"))       MP3Setup.EditorMode      = atoi(Value);
  else if (!strcasecmp(Name, "DisplayMode"))      MP3Setup.DisplayMode     = atoi(Value);
  else if (!strcasecmp(Name, "UseDeviceStillPicture"))       MP3Setup.UseDeviceStillPicture      = atoi(Value);
  else if (!strcasecmp(Name, "MenuMode"))         MP3Setup.MenuMode        = atoi(Value);
  else if (!strcasecmp(Name, "TargetLevel"))      MP3Setup.TargetLevel     = atoi(Value);
  else if (!strcasecmp(Name, "LimiterLevel"))     MP3Setup.LimiterLevel    = atoi(Value);
  else if (!strcasecmp(Name, "Only48kHz"))        MP3Setup.Only48kHz       = atoi(Value);
  else if (!strcasecmp(Name, "UseProxy"))         MP3Setup.UseProxy        = atoi(Value);
  else if (!strcasecmp(Name, "ProxyHost"))        strn0cpy(MP3Setup.ProxyHost,Value,MAX_HOSTNAME);
  else if (!strcasecmp(Name, "ProxyPort"))        MP3Setup.ProxyPort       = atoi(Value);
  else if (!strcasecmp(Name, "UseCddb"))          MP3Setup.UseCddb         = atoi(Value);
  else if (!strcasecmp(Name, "CddbHost"))         strn0cpy(MP3Setup.CddbHost,Value,MAX_HOSTNAME);
  else if (!strcasecmp(Name, "CddbPort"))         MP3Setup.CddbPort        = atoi(Value);
  else if (!strcasecmp(Name, "AbortAtEOL"))       MP3Setup.AbortAtEOL      = atoi(Value);
  else if (!strcasecmp(Name, "AudioOutMode")) {
    MP3Setup.AudioOutMode = atoi(Value);
#ifndef WITH_OSS
    if(MP3Setup.AudioOutMode==AUDIOOUTMODE_OSS) {
      esyslog("WARNING: AudioOutMode OSS not supported, falling back to DVB");
      MP3Setup.AudioOutMode=AUDIOOUTMODE_DVB;
      }
#endif
    }
//#if VDRVERSNUM >= 10307
//  else if (!strcasecmp(Name, "ReplayDisplay"))      	MP3Setup.ReplayDisplay = atoi(Value);
//#endif
  else if (!strcasecmp(Name, "HideMainMenu"))       	MP3Setup.HideMainMenu              = atoi(Value);
  else if (!strcasecmp(Name, "KeepSelect"))         	MP3Setup.KeepSelect                = atoi(Value);
  else if (!strcasecmp(Name, "ExitClose"))         	MP3Setup.ExitClose                 = atoi(Value);
  else if (!strcasecmp(Name, "AdminMode"))         	MP3Setup.AdminMode                 = atoi(Value);
  else if (!strcasecmp(Name, "EnableRating"))         	MP3Setup.EnableRating              = atoi(Value);
  else if (!strcasecmp(Name, "RateCompat"))         	strn0cpy(MP3Setup.RateCompat,Value,sizeof(MP3Setup.RateCompat));
  else if (!strcasecmp(Name, "RateInFile"))         	MP3Setup.RateInFile                = atoi(Value);
  else if (!strcasecmp(Name, "RatingFirst"))         	MP3Setup.RatingFirst               = atoi(Value);
  else if (!strcasecmp(Name, "Jumptime"))          	MP3Setup.Jumptime                  = atoi(Value);
  else if (!strcasecmp(Name, "CopyDir"))           	strn0cpy(MP3Setup.CopyDir,Value,sizeof(MP3Setup.CopyDir));
  else if (!strcasecmp(Name, "RecordDir"))           	strn0cpy(MP3Setup.RecordDir,Value,sizeof(MP3Setup.RecordDir));
  else if (!strcasecmp(Name, "RecordOpts"))           	strn0cpy(MP3Setup.RecordOpts,Value,sizeof(MP3Setup.RecordOpts));
  else if (!strcasecmp(Name, "CoverDir"))           	strn0cpy(MP3Setup.CoverDir,Value,sizeof(MP3Setup.CoverDir));
  else if (!strcasecmp(Name, "OSDoffsetx"))        	MP3Setup.OSDoffsetx                = atoi(Value);
  else if (!strcasecmp(Name, "OSDoffsety"))        	MP3Setup.OSDoffsety                = atoi(Value);
  else if (!strcasecmp(Name, "Rowcount"))        	MP3Setup.Rowcount                  = atoi(Value);
#ifdef HAVE_VDR_SPEZIAL
  else if (!strcasecmp(Name, "FullSize"))        	MP3Setup.FullSize                  = atoi(Value);
#endif
  else if (!strcasecmp(Name, "ArtistFirst"))            MP3Setup.ArtistFirst               = atoi(Value);
  else if (!strcasecmp(Name, "Coversize"))              MP3Setup.Coversize                 = atoi(Value);
  else if (!strcasecmp(Name, "ShowFileName"))           MP3Setup.ShowFileName              = atoi(Value);
  else if (!strcasecmp(Name, "MaxResult"))              MP3Setup.MaxResult                 = atoi(Value);
  else if (!strcasecmp(Name, "ImgAlpha"))               MP3Setup.ImgAlpha                  = atoi(Value);
  else if (!strcasecmp(Name, "ImgDither"))              MP3Setup.ImgDither                 = atoi(Value);
  else if (!strcasecmp(Name, "EnableSpectrumAnalyzer")) MP3Setup.EnableSpectrumAnalyzer    = atoi(Value);
  else if (!strcasecmp(Name, "ThemesSA"))                MP3Setup.ThemesSA                 = atoi(Value);
  else if (!strcasecmp(Name, "FalloffSA"))              MP3Setup.FalloffSA                 = atoi(Value);
#ifdef HAVE_MAGICK
  else if (!strcasecmp(Name, "MaxCacheFill"))           MP3Setup.MaxCacheFill              = atoi(Value);
  else if (!strcasecmp(Name, "ImgAspect"))              MP3Setup.ImgAspect                 = atoi(Value);
  else if (!strcasecmp(Name, "ImgTreeDepth"))           MP3Setup.ImgTreeDepth              = atoi(Value);
#endif
  else return false;
  return true;
}

cMenuSetupMP3::cMenuSetupMP3(void)
{
  static const char allowed[] = { "abcdefghijklmnopqrstuvwxyz0123456789-_@/" };


  sathemes[e1_SA8]      = tr("1 x 8 bars");
  sathemes[e1_SA16]     = tr("1 x 16 bars");
  sathemes[e1_SA32]     = tr("1 x 32 bars");
  sathemes[e1_SA64]     = tr("1 x 64 bars");
  sathemes[e2_SA4]      = tr("2 x 4 bars");
  sathemes[e2_SA8]      = tr("2 x 8 bars");
  sathemes[e2_SA16]     = tr("2 x 16 bars");
  sathemes[e2_SA32]     = tr("2 x 32 bars");
  sathemes[e2_SAVol]    = tr("Volume");
  int numModes=0;
  aout[numModes]=tr("DVB"); amodes[numModes]=AUDIOOUTMODE_DVB; numModes++;
#ifdef WITH_OSS
  aout[numModes]=tr("OSS"); amodes[numModes]=AUDIOOUTMODE_OSS; numModes++;
#endif
  amode=0;
  for(int i=0; i<numModes; i++)
    if(amodes[i]==MP3Setup.AudioOutMode) { amode=i; break; }

  SetSection(tr("MP3"));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Artist-Title in tracklist"),  &MP3Setup.ArtistFirst));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Admin mode"),                 &MP3Setup.AdminMode));
  Add(new cMenuEditIntItem( tr("Setup.MP3$Jump interval (FFW/FREW)"),   &MP3Setup.Jumptime,1,360));
  Add(new cMenuEditStrItem( tr("Setup.MP3$Where to copy tracks"),        MP3Setup.CopyDir,255,allowed));
  Add(new cMenuEditStrItem( tr("Setup.MP3$Where to record streams"),     MP3Setup.RecordDir,255,allowed));
  Add(new cMenuEditStrItem( tr("Setup.MP3$Recording options"),           MP3Setup.RecordOpts,255,allowed));
  Add(new cMenuEditStrItem( tr("Setup.MP3$Where are the artistcover"),   MP3Setup.CoverDir,255,allowed));
  Add(new cMenuEditIntItem(tr("Setup.MP3$OSD Offset X"),                &MP3Setup.OSDoffsetx,-50 ,50));
  Add(new cMenuEditIntItem(tr("Setup.MP3$OSD Offset Y"),                &MP3Setup.OSDoffsety,-50 ,50));
  coversize[0]=tr("normal");
  coversize[1]=tr("double");
  coversize[2]=("4:3");
  coversize[3]=tr("fullscreen");
  Add(new cMenuEditStraItem(tr("Setup.MP3$Coversize"),                  &MP3Setup.Coversize,4,coversize));
#ifdef HAVE_MAGICK
  Add(new cMenuEditIntItem(tr("Setup.MP3$Max. cache in MB for covers"), &MP3Setup.MaxCacheFill,1,1024));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Use Aspect ration for covers"),&MP3Setup.ImgAspect));
  Add(new cMenuEditIntItem(tr("Setup.MP3$Image quality >=slower"),      &MP3Setup.ImgTreeDepth,1 ,6));
#endif
  Add(new cMenuEditIntItem(tr("Setup.MP3$Transparency for covers"),     &MP3Setup.ImgAlpha,1,255));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Use Dithering for covers"),   &MP3Setup.ImgDither));

  Add(new cMenuEditBoolItem(tr("Setup.MP3$Enable spectrum analyzer"),   &MP3Setup.EnableSpectrumAnalyzer));
  Add(new cMenuEditStraItem(tr("Setup.MP3$Visualization"),              &MP3Setup.ThemesSA, eSAThemeMaxNumber, sathemes));
  Add(new cMenuEditIntItem(tr("Setup.MP3$Spectrum analyzer bar falloff value"),&MP3Setup.FalloffSA,  0, 20));

  Add(new cMenuEditBoolItem(tr("Setup.MP3$Enable rating"),              &MP3Setup.EnableRating));
  Add(new cMenuEditStrItem(tr("Setup.MP3$Email for rating"),            MP3Setup.RateCompat,255,allowed));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Store rating in file"),       &MP3Setup.RateInFile));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Rating as first red key"),    &MP3Setup.RatingFirst));
#ifdef HAVE_VDR_SPEZIAL
  Add(new cMenuEditIntItem(tr("Setup.MP3$Display tracks"),              &MP3Setup.Rowcount,4 ,9));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Use extensions for 4MB Mod"), &MP3Setup.FullSize));
#else
  Add(new cMenuEditIntItem(tr("Setup.MP3$Display tracks"),              &MP3Setup.Rowcount,4 ,8));
#endif
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Show filename in search (faster)"),   &MP3Setup.ShowFileName));
  Add(new cMenuEditIntItem(tr("Setup.MP3$Max. results while search"),   &MP3Setup.MaxResult,1 ,50000));
  Add(new cMenuEditStraItem(tr("Setup.MP3$Audio output mode"),          &amode,numModes,aout));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Audio mode"),                 &MP3Setup.AudioMode, tr("Round"), tr("Dither")));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Use 48kHz mode only"),        &MP3Setup.Only48kHz));
  Add(new cMenuEditIntItem( tr("Setup.MP3$Display mode"),               &MP3Setup.DisplayMode, 1, 4));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Use DeviceStillPicture"),     &MP3Setup.UseDeviceStillPicture));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Initial loop mode"),          &MP3Setup.InitLoopMode));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Initial shuffle mode"),       &MP3Setup.InitShuffleMode));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Abort player at end of list"),&MP3Setup.AbortAtEOL));
  scan[0]=tr("disabled");
  scan[1]=tr("ID3 only");
  scan[2]=tr("ID3 & Level");
  Add(new cMenuEditStraItem(tr("Setup.MP3$Background scan"),       &MP3Setup.BgrScan, 3, scan));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Editor display mode"),   &MP3Setup.EditorMode, tr("Filenames"), tr("ID3 names")));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Mainmenu mode"),         &MP3Setup.MenuMode, tr("Playlists"), tr("Browser")));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Keep selection menu"),   &MP3Setup.KeepSelect));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Exit stop playback"),    &MP3Setup.ExitClose));
  Add(new cMenuEditBoolItem(tr("Hide mainmenu entry"),             &MP3Setup.HideMainMenu));
  Add(new cMenuEditIntItem( tr("Setup.MP3$Normalizer level"),      &MP3Setup.TargetLevel, 0, MAX_TARGET_LEVEL));
  Add(new cMenuEditIntItem( tr("Setup.MP3$Limiter level"),         &MP3Setup.LimiterLevel, MIN_LIMITER_LEVEL, 100));
  Add(new cMenuEditBoolItem(tr("Setup.MP3$Use HTTP proxy"),        &MP3Setup.UseProxy));
  Add(new cMenuEditStrItem( tr("Setup.MP3$HTTP proxy host"),       MP3Setup.ProxyHost,MAX_HOSTNAME,allowed));
  Add(new cMenuEditIntItem( tr("Setup.MP3$HTTP proxy port"),       &MP3Setup.ProxyPort,1,65535));
  cddb[0]=tr("disabled");
  cddb[1]=tr("local only");
  cddb[2]=tr("local&remote");
  Add(new cMenuEditStraItem(tr("Setup.MP3$CDDB for CD-Audio"),     &MP3Setup.UseCddb,3,cddb));
  Add(new cMenuEditStrItem( tr("Setup.MP3$CDDB server"),           MP3Setup.CddbHost,MAX_HOSTNAME,allowed));
  Add(new cMenuEditIntItem( tr("Setup.MP3$CDDB port"),             &MP3Setup.CddbPort,1,65535));
  
}

void cMenuSetupMP3::Store(void)
{
  MP3Setup.AudioOutMode=amodes[amode];

  SetupStore("InitLoopMode",     MP3Setup.InitLoopMode   );
  SetupStore("InitShuffleMode",  MP3Setup.InitShuffleMode);
  SetupStore("AudioMode",        MP3Setup.AudioMode      );
  SetupStore("AudioOutMode",     MP3Setup.AudioOutMode   );
  SetupStore("BgrScan",          MP3Setup.BgrScan        );
  SetupStore("EditorMode",       MP3Setup.EditorMode     );
  SetupStore("DisplayMode",      MP3Setup.DisplayMode    );
  SetupStore("UseDeviceStillPicture", MP3Setup.UseDeviceStillPicture );
  SetupStore("MenuMode",         MP3Setup.MenuMode       );
  SetupStore("TargetLevel",      MP3Setup.TargetLevel    );
  SetupStore("LimiterLevel",     MP3Setup.LimiterLevel   );
  SetupStore("Only48kHz",        MP3Setup.Only48kHz      );
  SetupStore("UseProxy",         MP3Setup.UseProxy       );
  SetupStore("ProxyHost",        MP3Setup.ProxyHost      );
  SetupStore("ProxyPort",        MP3Setup.ProxyPort      );
  SetupStore("UseCddb",          MP3Setup.UseCddb        );
  SetupStore("CddbHost",         MP3Setup.CddbHost       );
  SetupStore("CddbPort",         MP3Setup.CddbPort       );
  SetupStore("AbortAtEOL",       MP3Setup.AbortAtEOL     );
  SetupStore("HideMainMenu",     MP3Setup.HideMainMenu   );
  SetupStore("KeepSelect",       MP3Setup.KeepSelect     );
  SetupStore("ExitClose",        MP3Setup.ExitClose      );
  SetupStore("AdminMode",        MP3Setup.AdminMode      );
  SetupStore("EnableRating",     MP3Setup.EnableRating   );
  SetupStore("RateCompat",       MP3Setup.RateCompat     );
  SetupStore("RateInFile",       MP3Setup.RateInFile     );
  SetupStore("RatingFirst",      MP3Setup.RatingFirst    );
  SetupStore("Jumptime",         MP3Setup.Jumptime       );
  SetupStore("CopyDir",          MP3Setup.CopyDir        );
  SetupStore("RecordDir",        MP3Setup.RecordDir      );
  SetupStore("RecordOpts",       MP3Setup.RecordOpts     );
  SetupStore("CoverDir",         MP3Setup.CoverDir       );
  SetupStore("OSDoffsetx",       MP3Setup.OSDoffsetx     );
  SetupStore("OSDoffsety",       MP3Setup.OSDoffsety     );
  SetupStore("Rowcount",         MP3Setup.Rowcount       );
#ifdef HAVE_VDR_SPEZIAL
  SetupStore("FullSize",         MP3Setup.FullSize       );
#endif
  SetupStore("ArtistFirst",      MP3Setup.ArtistFirst    );
  SetupStore("Coversize",        MP3Setup.Coversize      );
  SetupStore("ShowFileName",     MP3Setup.ShowFileName   );
  SetupStore("MaxResult",        MP3Setup.MaxResult      );
  SetupStore("ImgAlpha",         MP3Setup.ImgAlpha       );
  SetupStore("ImgDither",        MP3Setup.ImgDither      );
  SetupStore("EnableSpectrumAnalyzer", MP3Setup.EnableSpectrumAnalyzer);
  SetupStore("ThemesSA",         MP3Setup.ThemesSA);
  SetupStore("FalloffSA",        MP3Setup.FalloffSA);
#ifdef HAVE_MAGICK
  SetupStore("MaxCacheFill",     MP3Setup.MaxCacheFill   );
  SetupStore("ImgAspect",        MP3Setup.ImgAspect      );
  SetupStore("ImgTreeDepth",     MP3Setup.ImgTreeDepth   );
#endif
}

VDRPLUGINCREATOR(cPluginMp3); // Don't touch this!
