#include "texteffects.h"
#include "common.h"
#include "config.h"

#ifndef DISABLE_ANIMATED_TEXT
//Redefine macros
#undef  TE_LOCK
#define TE_LOCK   UpdateLock()
#undef  TE_UNLOCK
#define TE_UNLOCK UpdateUnlock()
#endif

#ifdef DISABLE_ANIMATED_TEXT
cEnigmaTextEffects EnigmaTextEffects;

cEnigmaTextEffects::cEnigmaTextEffects(void) : osd(NULL)
{}

#else
cEnigmaTextEffects EnigmaTextEffects("EnigmaNG effects");

cEnigmaTextEffects::cEnigmaTextEffects(const char *Description) : cThread(Description), osd(NULL), condSleep(), mutexSleep()
{
//  SetPriority(19);
}
#endif

cEnigmaTextEffects::~cEnigmaTextEffects(void)
{
#ifndef DISABLE_ANIMATED_TEXT
  Stop();
#endif
}

#ifndef DISABLE_ANIMATED_TEXT
void cEnigmaTextEffects::Action(void)
{
  debug("cEnigmaTextEffects::Action() %p\n", pthread_self());

  mutexSleep.Lock();

  while (EnigmaConfig.useTextEffects && osd && Running()) {
    uint64_t nNow = cTimeMs::Now();
    int nSleepMs = 0;

    TE_LOCK;
    for (vector<tEffect*>::iterator effect = vecEffects.begin(); (effect != vecEffects.end()) && Running() && osd; effect++) {
      tEffect *e = (*effect);
      if (e == NULL)
        continue;

      if (e->nNextUpdate == 0) {
        e->nNextUpdate = nNow + (e->nAction == 0 ? EnigmaConfig.scrollPause : EnigmaConfig.blinkPause);
      } else if(nNow >= e->nNextUpdate) {
        DoEffect(e, nNow);
      }

      int nDiff = max(3, (int)(e->nNextUpdate - nNow));
      if (nSleepMs == 0 || nDiff < nSleepMs)
        nSleepMs = nDiff;
    }

    if (osd)
      osd->Flush();
    TE_UNLOCK;

    if (nSleepMs)
      condSleep.TimedWait(mutexSleep, nSleepMs);
    else
      condSleep.Wait(mutexSleep);
  }

  mutexSleep.Unlock();
}

void cEnigmaTextEffects::DoEffect(tEffect *e, uint64_t nNow)
{
  switch (e->nAction) {
    case 0:  // Scroll
      DoScroll(e, nNow);
      break;

    case 1: // Blink
      DoBlink(e, nNow);
      break;
  }
}

void cEnigmaTextEffects::DoScroll(tEffect *e, uint64_t nNow)
{
//  debug("cEnigmaTextEffects::DoScroll()\n");
  if (e->Font->Width(e->strText.c_str()) <= e->Width) {
    if (e->Skin)
      e->Skin->DrawTitle(e->strText.c_str());
    else
      osd->DrawText(e->x, e->y, e->strText.c_str(), e->ColorFg, e->ColorBg, e->Font, e->Width, e->Height, e->Alignment);

    if (nNow)
      e->nNextUpdate = nNow + EnigmaConfig.scrollPause;
    return;
  }

  if (nNow) {
    int nDelay = EnigmaConfig.scrollDelay;
    switch (e->nDirection) {
      case 0: // Scroll from left to right
        if (e->Font->Width(e->strText.c_str() + e->nOffset) <= e->Width) {
          if (EnigmaConfig.scrollMode)
            e->nDirection = 2;
          else
            e->nDirection = 1;
          nDelay = EnigmaConfig.scrollPause;
        } else if (e->nOffset < e->strText.length())
          e->nOffset++;
        break;

      case 1: // Scroll from right to left
        if (e->nOffset > 0)
          e->nOffset--;
        if (e->nOffset <= 0) {
          e->nDirection = false;
          nDelay = EnigmaConfig.scrollPause;
        }
        break;

      case 2: // Restart scrolling from the left
        nDelay = EnigmaConfig.scrollPause;
        e->nOffset = 0;
        e->nDirection = 0;
        break;
    }
    e->nNextUpdate = nNow + nDelay;
  }
//  printf("SCROLL: %d %d %d/%d (%s) %d %lu %lu\n", e->nOffset, e->nDirection, e->Font->Width(e->strText.c_str() + e->nOffset), e->Width, e->strText.c_str() + e->nOffset, e->strText.length(), nNow, e->nNextUpdate);
  if (e->Skin)
    e->Skin->DrawTitle(e->strText.c_str() + e->nOffset);
  else
    osd->DrawText(e->x, e->y, e->strText.c_str() + e->nOffset, e->ColorFg, e->ColorBg, e->Font, e->Width, e->Height);
}

void cEnigmaTextEffects::DoBlink(tEffect *e, uint64_t nNow)
{
//  debug("cEnigmaTextEffects::DoBlink()\n");
  if (nNow) {
    e->nDirection = (e->nDirection == 0 ? 1 : 0);
    e->nNextUpdate = nNow + EnigmaConfig.blinkPause;
  }
  if (e->nDirection == 1)
    osd->DrawText(e->x, e->y, e->strText.c_str() + e->nOffset, e->ColorFg, e->ColorBg, e->Font, e->Width, e->Height, e->Alignment);
  else
    osd->DrawText(e->x, e->y, e->strText.c_str() + e->nOffset, e->ColorBg, e->ColorBg, e->Font, e->Width, e->Height, e->Alignment);
}

bool cEnigmaTextEffects::Start(cOsd *o)
{
  osd = o;

  if (!EnigmaConfig.useTextEffects)
    return false;

  debug("cEnigmaTextEffects::Start(%p) %p\n", osd, pthread_self());

  if (osd == NULL)
    return false;

  if (Running()) {
    error("cEnigmaTextEffects::Start - already running\n");
    return false;
  }

  TE_LOCK;
  return cThread::Start();
}

void cEnigmaTextEffects::Stop(void)
{
  debug("cEnigmaTextEffects::Stop()\n");
  TE_LOCK;
  osd = NULL;
  Clear();
  TE_UNLOCK;
  Wakeup(); // force update
}

void cEnigmaTextEffects::Clear(void)
{
  debug("cEnigmaTextEffects::Clear()\n");

  //Must be TE_LOCKed by caller

  for (vector<tEffect*>::iterator effect = vecEffects.begin(); effect != vecEffects.end(); effect++) {
    delete(*effect);
  }

  vecEffects.clear();
}

void cEnigmaTextEffects::ResetText(int i, tColor ColorFg, tColor ColorBg)
{
  debug("cEnigmaTextEffects::ResetText(%d)\n", i);

  //Must be TE_LOCKed by caller

  if (i < 0 || i >= (int)vecEffects.size())
    return;

  tEffect *e = vecEffects[i];
  if (e) {
    osd->DrawText(e->x, e->y, e->strText.c_str(),
                  ColorFg ? ColorFg : e->ColorFg,
                  ColorBg ? ColorBg : e->ColorBg,
                  e->Font, e->Width, e->Height);
    delete(e);
    vecEffects[i] = NULL;
  }
}

void cEnigmaTextEffects::UpdateTextWidth(int i, int Width)
{
  debug("cEnigmaTextEffects::UpdateTextWidth(%d)\n", i);

  //Must be TE_LOCKed by caller

  if (i < 0 || i >= (int)vecEffects.size())
    return;

  tEffect *e = vecEffects[i];
  if (e) {
    e->Width = Width;
  }
}

int cEnigmaTextEffects::DrawAnimatedTitle(int o_id, int action, const char *s, const cFont *Font, int Width, cSkinEnigmaOsd *skin)
{
  //Must be TE_LOCKed by caller

  if (Font == NULL || osd == NULL)
    return -1;

  debug("cEnigmaTextEffects::DrawAnimatedTitle(%d, %d, %s)\n", o_id, EnigmaConfig.useTextEffects, s);

  if (o_id >= 0) {
    // Update animated text
    tEffect *effect = vecEffects[o_id];
    if (effect) {
      if (s == NULL)
        effect->strText = "";
      else if (strcmp(effect->strText.c_str(), s) != 0) {
        effect->strText = s;
        effect->nOffset = 0;
        effect->nDirection = 0;
      }
      DoEffect(effect);
      return o_id;
    } else {
      return -1;
    }
  } else {
    skin->DrawTitle(s);
    if (EnigmaConfig.useTextEffects && ((Font->Width(s ? s : "") > Width) || (action > 0))) {
      // New scrolling text
      tEffect *effect = new tEffect;
      if (effect == NULL) {
        return -1;
      }

      effect->nAction = action;
      effect->strText = string(s ? s : "");
      effect->Width = Width;
      effect->Font = Font;
      effect->Skin = skin;
      vecEffects.push_back(effect);
      int id = vecEffects.size() - 1;
      return id;
    } else {
      return -1;
    }
  }
}

int cEnigmaTextEffects::DrawAnimatedText(int o_id, int action, int x, int y, const char *s, tColor ColorFg, tColor ColorBg, const cFont *Font, int Width, int Height, int Alignment)
{
  //Must be TE_LOCKed by caller

  if (Font == NULL || osd == NULL)
    return -1;

  debug("cEnigmaTextEffects::DrawAnimatedText(%d, %d, %s)\n", o_id, EnigmaConfig.useTextEffects, s);

  if (o_id >= 0) {
    // Update animated text
    tEffect *effect = vecEffects[o_id];
    if (effect) {
      if (s == NULL)
        effect->strText = "";
      else if (strcmp(effect->strText.c_str(), s) != 0) {
        effect->strText = s;
        effect->nOffset = 0;
        effect->nDirection = 0;
      }
      DoEffect(effect);
      return o_id;
    } else {
      return -1;
    }
  } else {
    osd->DrawText(x, y, s ? s : "", ColorFg, ColorBg, Font, Width, Height, Alignment);
    // New animated text
    tEffect *effect = new tEffect;
    if (effect == NULL) {
      return -1;
    }

    effect->nAction = action;
    effect->strText = string(s ? s : "");
    effect->x = x;
    effect->y = y;
    effect->Width = Width;
    effect->Height = Height;
    effect->ColorFg = ColorFg;
    effect->ColorBg = ColorBg;
    effect->Font = Font;
    effect->Alignment = Alignment;
    vecEffects.push_back(effect);
    int id = vecEffects.size() - 1;
    return id;
  }
}
#endif //DISABLE_ANIMATED_TEXT

// vim:et:sw=2:ts=2:
