/*
 * avards.c: A plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id$
 */

#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <linux/dvb/video.h>
#include <linux/videodev.h>
#include <linux/dvb/frontend.h>

#include <vdr/tools.h>
#include <vdr/osdbase.h>

#include "detector.h"
#include "avards.h"

struct SAspectInfo
{
  float       fMinAspect;
  const char* pszName;
  unsigned    iWssData;
};

SAspectInfo aAspectInfo[] =
{
  { 1.46,    "4:3",    0x08 },
  { 1.66,    "L14:9",  0x01 },
//{ 1.77,    "16:9",   0x07 },
  { 1.90,    "L16:9",  0x0b },
  { FLT_MAX, "L>16:9", 0x0d }
};



cDetector::cDetector(void)
:cThread("avards-detector")
{
  snprintf(pszDvbVideo, MAXSTRING, "%s%s", config.pszDvb, "/video0");
  snprintf(pszDvbFrontend, MAXSTRING, "%s%s", config.pszDvb, "/frontend0");

  naAspectInfo = sizeof(aAspectInfo) / sizeof(SAspectInfo);

  //avoid overzoom (-o)
  naAspectInfo--;
  aAspectInfo[naAspectInfo-1].fMinAspect = aAspectInfo[naAspectInfo].fMinAspect;
}

cDetector::~cDetector()
{
  Stop();
}

bool cDetector::GetOSDSize(avards_OSDsize_v1_0 *OSDsize)
{
  //isyslog("avards: cDetector::GetOSDSize()");
  OSDsize->left   = Setup.OSDLeft;
  OSDsize->width  = Setup.OSDWidth;

  if (!Active()) {
    OSDsize->top    = Setup.OSDTop;
    OSDsize->height = Setup.OSDHeight;
  } else {
    OSDsize->top    = Setup.OSDTop - 1;
    OSDsize->height = Setup.OSDHeight - 1;
  }

  return true;
}

bool cDetector::StartUp()
{
  if (!Running())
    Start();
  return true;
}

void cDetector::Stop()
{
  if (Running()) {
    Cancel(-1);
    Wait.Signal();
  }
}

void cDetector::Action(void)
{
  int devDvbVideo = open(pszDvbVideo,O_RDONLY|O_NONBLOCK);
  if (devDvbVideo < 0)
    esyslog("avards: Error: Can't open dvb video device '%s' (%s)\n", pszDvbVideo, strerror(errno));
  else
  {
    int devDvbFrontend = 0;
    if (config.FrontendHasLockTest)
      devDvbFrontend = open(pszDvbFrontend,O_RDONLY|O_NONBLOCK);
    if (config.FrontendHasLockTest && (devDvbFrontend < 0))
      esyslog("avards: Error: Can't open dvb frontend device '%s' (%s)\n", pszDvbFrontend, strerror(errno));
    else
    {
      int devVideo = open(config.pszDevVideo, O_RDWR);
      if (devVideo < 0)
        esyslog("avards: Error: Can't open video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
      else
      {
        wssoverdrive = new cWSSoverdrive;
        if (wssoverdrive)
        {
          wssoverdrive->OpenVbiDevice(config.pszDevVbi);
          struct video_mbuf mbufVideo;
          if (ioctl(devVideo, VIDIOCGMBUF, &mbufVideo) != 0) 
            esyslog("avards: Error: Can't open video buffer on video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
          else
          {
            unsigned char *pbVideo = (unsigned char *)mmap(0, mbufVideo.size, PROT_READ | PROT_WRITE, MAP_SHARED, devVideo, 0);
            if (!pbVideo || (pbVideo == (unsigned char *)-1))
              esyslog("avards: Error: Can't map video buffer of video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
            else
            {
              struct video_capability capabilityVideo;
              if (ioctl(devVideo, VIDIOCGCAP, &capabilityVideo))
                esyslog("avards: Error: Can't query caps of video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
              else
              {
                struct video_mmap mmapVideo;
                mmapVideo.frame = 0;
                mmapVideo.width = capabilityVideo.maxwidth;
                mmapVideo.height = capabilityVideo.maxheight;
                mmapVideo.format = VIDEO_PALETTE_GREY;
                int bAnamorph = 0;
                int iAspectInfo = -1;
                char** ppszDelay = (char**)calloc(config.Delay, sizeof(char*));
                for (int i = 0; i < config.Delay; i++)
                  ppszDelay[i] = strdup("");
                int bDelay = 1;
                int bError = 0;
                while (!bError && Running())
                {
                  int bFormatChanged = 0;
                  fe_status_t FrontendStatus;
                  if (config.FrontendHasLockTest && (bError = (ioctl(devDvbFrontend, FE_READ_STATUS, &FrontendStatus) < 0)))
                    esyslog("avards: Error: Can't get status of device '%s' (%s)\n", pszDvbFrontend, strerror(errno));
                  else
                  {
                    if (!config.FrontendHasLockTest || (FrontendStatus & FE_HAS_LOCK))
                    {
                      struct video_status statusVideo;
                      bError = ioctl(devDvbVideo, VIDEO_GET_STATUS, &statusVideo) < 0;
                      if (bError)
                        esyslog("avards: Error: Can't get status of device '%s' (%s)\n", pszDvbVideo, strerror(errno));
                      else
                      {
                        switch (statusVideo.video_format)
                        {
                        case VIDEO_FORMAT_16_9:
                        case VIDEO_FORMAT_221_1:
                          if (!bAnamorph)
                          {
                            bFormatChanged = 1;
                            bAnamorph = 1;
                          }
                          break;
                        case  VIDEO_FORMAT_4_3:
                        default:
                          if (bAnamorph)
                          {
                            iAspectInfo = -1;
                            bAnamorph = 0;
                          }
                          bError = ioctl(devVideo, VIDIOCMCAPTURE, &mmapVideo) | ioctl(devVideo, VIDIOCSYNC, &mmapVideo.frame);
                          //if (bError = (  ioctl(devVideo, VIDIOCMCAPTURE, &mmapVideo)
                          //  | ioctl(devVideo, VIDIOCSYNC, &mmapVideo.frame)))
                          if (bError)
                            esyslog("avards: Error: Can't grab on video device '%s' (%s)\n", config.pszDevVideo, strerror(errno));
                          else
                          {
                            const int nMaxLogoWidth = int(mmapVideo.width*config.MaxLogoWidth_pct/100);
                            const int nYoverscan = int (mmapVideo.height*config.Overscan_pct/100);
                            const int nXoverscan = int (mmapVideo.width*config.Overscan_pct/100);

                            unsigned char iPanToneMin;
                            unsigned char iPanToneMax;

                            int nActiveLines = 0;
                            int iYtop = nYoverscan;

                            if (config.PanToneBlackMax)
                            {
                              iPanToneMin = 0;
                              iPanToneMax = config.PanToneBlackMax;
                            }
                            else
                            {
                              int aHistogram[256];
                              memset(aHistogram, 0, sizeof(aHistogram));
                              unsigned char *pbVideoLine = pbVideo+iYtop*mmapVideo.width;
                              for (int iX = nXoverscan; iX < mmapVideo.width-nXoverscan; iX++)
                                aHistogram[pbVideoLine[iX]]++;
                              int iXmax = 0;
                              for (int i = 0; i < 256; i++)
                                if (aHistogram[i] > aHistogram[iXmax])
                                  iXmax = i;
                                if (iXmax - config.PanToneTolerance < 0)
                                  iPanToneMin = 0;
                                else
                                  iPanToneMin = iXmax - config.PanToneTolerance;
                                if (iXmax + config.PanToneTolerance > 255)
                                  iPanToneMax = 255;
                                else
                                  iPanToneMax = iXmax + config.PanToneTolerance;
                            }

                            while (iYtop < mmapVideo.height/2)
                            {
                              unsigned char *pbVideoLine = pbVideo+iYtop*mmapVideo.width;
                              int iXleft = nXoverscan;
                              while ( (iXleft < (mmapVideo.width-nXoverscan)) &&
                                      (pbVideoLine[iXleft] >= iPanToneMin)    &&
                                      (pbVideoLine[iXleft] <= iPanToneMax)  )
                                iXleft++;

                              int iXright = mmapVideo.width-nXoverscan-1;
                              while ( (iXright > iXleft) &&
                                      (pbVideoLine[iXright] >= iPanToneMin) &&
                                      (pbVideoLine[iXright] <= iPanToneMax) )
                                iXright--;

                              if (iXright - iXleft > nMaxLogoWidth)
                                break;
                              iYtop++;
                            }

                            if (iYtop < mmapVideo.height/2)
                            {
                              int iYbottom = nYoverscan;
                              while (iYbottom < mmapVideo.height/2)
                              {
                                unsigned char *pbVideoLine = pbVideo+(mmapVideo.height-1-iYbottom)*mmapVideo.width;
                                int iXleft = nXoverscan;
                                while ( (iXleft < (mmapVideo.width-nXoverscan)) &&
                                        (pbVideoLine[iXleft] >= iPanToneMin) &&
                                        (pbVideoLine[iXleft] <= iPanToneMax)  )
                                  iXleft++;

                                int iXright = mmapVideo.width-nXoverscan-1;
                                while ( (iXright > iXleft) &&
                                        (pbVideoLine[iXright] >= iPanToneMin) &&
                                        (pbVideoLine[iXright] <= iPanToneMax) )
                                  iXright--;

                                if (iXright - iXleft > nMaxLogoWidth)
                                  break;
                                iYbottom++;
                              }

                              if (iYbottom < mmapVideo.height/2)
                              {
                                int iBlackLines;
                                if (iYtop > iYbottom)
                                  iBlackLines = 2*iYbottom;
                                else
                                  iBlackLines = 2*iYtop;
                                nActiveLines = mmapVideo.height - iBlackLines;
                              }
                            }

                            if (nActiveLines > 0)
                            {
                              const float fLinearWidth = 4./3.*mmapVideo.height;
                              if (iAspectInfo < 0)
                                bFormatChanged = 1;
                              else
                              {
                                const int nAspectHyst = 2;
                                int nMinActiveLines = (int)(fLinearWidth/aAspectInfo[iAspectInfo].fMinAspect) - nAspectHyst;
                                int nMaxActiveLines =   (iAspectInfo > 0)
                                  ? (int)(fLinearWidth/aAspectInfo[iAspectInfo-1].fMinAspect) + nAspectHyst
                                  : mmapVideo.height;
                                bFormatChanged = (nActiveLines < nMinActiveLines) || (nActiveLines > nMaxActiveLines); 
                              }
                              float fAspect = fLinearWidth/nActiveLines;
                              if (bFormatChanged)
                              {
                                iAspectInfo = 0;
                                while (fAspect > aAspectInfo[iAspectInfo].fMinAspect)
                                  iAspectInfo++;
                              }
                            }
                          }
                          break;
                        }
                      }
                    }
                  }
                  if (!bError)
                  {
                    if (bAnamorph)
                    {
                      if (bFormatChanged)
                      {
                        //printf("16:9\n");
                        //fflush(stdout);
                        isyslog("avards: switching to 16:9");
                        wssoverdrive->SetWSS(0x07);
                      }
                      free(ppszDelay[0]);
                      ppszDelay[0] = strdup("");
                      bDelay = 1;
                    }
                    else if (iAspectInfo >= 0)
                    {
                      static char szOutput[80];
                      if (bFormatChanged)
                      {
                        const char* pszAspect = aAspectInfo[iAspectInfo].pszName;
                        sprintf(szOutput, "%s", pszAspect);
                      }
                      int iDelay = 0;
                      if (!config.Delay || !strcmp(ppszDelay[0], ""))
                      {
                        iDelay = config.Delay;
                        bDelay = 1;
                        for (int i = 0; i < config.Delay; i++)
                        {
                          free(ppszDelay[i]);
                          ppszDelay[i] = strdup(szOutput);
                        }
                      }
                      else
                        while ((iDelay < config.Delay) && !strcmp(ppszDelay[iDelay], szOutput))
                          iDelay++;
                        if (iDelay < config.Delay)
                          bDelay = 1;
                        else if (bDelay)
                        {
                          bDelay = 0;
                          //printf("%s\n", szOutput);
                          //fflush(stdout);
                          isyslog("avards: switching to %s", szOutput);
                          wssoverdrive->SetWSS(aAspectInfo[iAspectInfo].iWssData);
                        }
                        if (config.Delay)
                        {
                          free(ppszDelay[config.Delay-1]);
                          memmove(ppszDelay+1, ppszDelay, sizeof(char*)*(config.Delay-1));
                          *ppszDelay = strdup(szOutput);
                        }
                    }

                    if (Running())
                      Wait.Wait(max(80, config.PollRate));
                  }
                }
                munmap(pbVideo, mbufVideo.size);
              }
            }
          }
        }
        wssoverdrive->CloseVbiDevice();
        delete wssoverdrive;
        wssoverdrive = NULL;
        close(devVideo);
      }
      if (config.FrontendHasLockTest)
        close(devDvbFrontend);
    }
    close(devDvbVideo);
  }
}




cWSSoverdrive::cWSSoverdrive(void)
{
  vbi_device = 0;
}

cWSSoverdrive::~cWSSoverdrive(void)
{
  if (vbi_device)
    CloseVbiDevice();
}

bool cWSSoverdrive::OpenVbiDevice(const char* szDevice)
{
  vbi_device = open(szDevice, O_WRONLY);
  if (vbi_device < 0) 
  {
    esyslog("avards: Error: Can't open vbi device '%s' (%s)\n", szDevice, strerror(errno));
    return false;
  }

  struct v4l2_format fmt;
  memset(&fmt, 0, sizeof fmt);
  fmt.type = V4L2_BUF_TYPE_SLICED_VBI_OUTPUT;
  fmt.fmt.sliced.service_set = V4L2_SLICED_WSS_625;
  if (ioctl(vbi_device, VIDIOC_S_FMT, &fmt) < 0) {
    esyslog("avards Error: Can't init vbi device '%s' (%s)\n", szDevice, strerror(errno));
    CloseVbiDevice();
    return false;
  }

  isyslog("avards: vbi device '%s' opened (%s)\n", szDevice, strerror(errno));
  return true;
}

void cWSSoverdrive::CloseVbiDevice(void)
{
  isyslog("avards: closing vbi device\n");
  if (vbi_device) {
    close(vbi_device);
    vbi_device = 0;
  }
}

void cWSSoverdrive::SetWSS(int iWssData)
{
  if (vbi_device) {
    isyslog("avards: setting WSS data to %02x", iWssData);
    struct v4l2_sliced_vbi_data data;
    memset(&data, 0, sizeof data);
    data.id = (iWssData) ? V4L2_SLICED_WSS_625 : 0;
    data.line = 23;
    data.data[0] = iWssData & 0xff;
    data.data[1] = (iWssData >> 8) & 0x3f;
    if (write(vbi_device, &data, sizeof data) != sizeof data) {
      esyslog("avards: Error: Can't write to device (%s)\n", strerror(errno));
      CloseVbiDevice();
    }
  }
}
