/*
 * protocolfile.c: IPTV plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id: protocolfile.c,v 1.14 2007/10/20 23:25:14 ajhseppa Exp $
 */

#include <fcntl.h>
#include <unistd.h>

#include <vdr/device.h>

#include "common.h"
#include "config.h"
#include "protocolfile.h"

cIptvProtocolFile::cIptvProtocolFile()
: fileDelay(0),
  readBufferLen(TS_SIZE * IptvConfig.GetReadBufferTsCount()),
  isActive(false)
{
  debug("cIptvProtocolFile::cIptvProtocolFile()\n");
  fileLocation = strdup("");
  // Allocate receive buffer
  readBuffer = MALLOC(unsigned char, readBufferLen);
  if (!readBuffer)
     error("ERROR: MALLOC() failed in ProtocolFile()");
}

cIptvProtocolFile::~cIptvProtocolFile()
{
  debug("cIptvProtocolFile::~cIptvProtocolFile()\n");
  // Drop open handles
  Close();
  // Free allocated memory
  free(fileLocation);
  free(readBuffer);
}

bool cIptvProtocolFile::OpenFile(void)
{
  debug("cIptvProtocolFile::OpenFile()\n");
  // Check that stream address is valid
  if (!isActive && !isempty(fileLocation)) {
     fileStream = fopen(fileLocation, "rb");
     ERROR_IF_RET(!fileStream || ferror(fileStream), "fopen()", return false);
     // Update active flag
     isActive = true;
     }
  return true;
}

void cIptvProtocolFile::CloseFile(void)
{
  debug("cIptvProtocolFile::CloseFile()\n");
  // Check that file stream is valid
  if (isActive && !isempty(fileLocation)) {
     fclose(fileStream);
     // Update active flag
     isActive = false;
     }
}

int cIptvProtocolFile::Read(unsigned char* *BufferAddr)
{
   //debug("cIptvProtocolFile::Read()\n");
   *BufferAddr = readBuffer;
   // Check errors
   if (ferror(fileStream)) {
      debug("Read error\n");
      return -1;
      }
   // Rewind if EOF
   if (feof(fileStream))
      rewind(fileStream);
   // Sleep before reading the file stream to prevent aggressive busy looping
   // and prevent transfer ringbuffer overflows
   if (fileDelay)
      cCondWait::SleepMs(fileDelay);
   // This check is to prevent a race condition where file may be switched off
   // during the sleep and buffers are disposed. Check here that the plugin is
   // still active before accessing the buffers
   if (isActive)
      return fread(readBuffer, sizeof(unsigned char), readBufferLen, fileStream);
   return -1;
}

bool cIptvProtocolFile::Open(void)
{
  debug("cIptvProtocolFile::Open()\n");
  // Open the file stream
  OpenFile();
  return true;
}

bool cIptvProtocolFile::Close(void)
{
  debug("cIptvProtocolFile::Close()\n");
  // Close the file stream
  CloseFile();
  return true;
}

bool cIptvProtocolFile::Set(const char* Location, const int Parameter, const int Index)
{
  debug("cIptvProtocolFile::Set(): Location=%s Parameter=%d Index=%d\n", Location, Parameter, Index);
  if (!isempty(Location)) {
     // Close the file stream
     CloseFile();
     // Update stream address and port
     fileLocation = strcpyrealloc(fileLocation, Location);
     fileDelay = Parameter;
     // Open the file for input
     OpenFile();
     }
  return true;
}

cString cIptvProtocolFile::GetInformation(void)
{
  //debug("cIptvProtocolFile::GetInformation()");
  return cString::sprintf("file://%s:%d", fileLocation, fileDelay);
}
