/*
 * cdstatus.c
 * watch cd/dvd drive status, especially the eject button of a locked drive
 * parts of the code were taken from auto-eject-cdrom.c:
 * Copyright (C) 2001 Jens Axboe <axboe@suse.de>
 * Modified and subsequently
 *   Copyright (C) 2005 Peter Willis <psyphreak@phreaker.net>
 *   (still no license on the thing as the original author included no license;
 *   maybe i should email him for permission?)
 */
#include <stdlib.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mount.h>

#include <vdr/thread.h>
#include <vdr/tools.h>
#include <vdr/plugin.h>
#if VDRVERSNUM >= 10332
#include <vdr/remote.h>
#endif
#include <vdr/recording.h>

#include <linux/cdrom.h>

#include <scsi/scsi.h>
//#include <scsi/sg.h>
#include <scsi/scsi_ioctl.h>

#include "cdstatus.h"

int debug = true;

#if VDRVERSNUM < 10332
cOsdObject *osd = NULL;
#endif

#define PROC_MOUNTS "/proc/mounts"

static char *GetMountPoint(const char *cd_device)
{
  FILE   *mtab;
  char   *line = NULL;
  size_t  lineLen = 0;
  char   *target = NULL;

  if ( (mtab = fopen(PROC_MOUNTS, "r")) == NULL )
  {
    esyslog("ERROR: couldn't open %s: %m", PROC_MOUNTS);
    return(NULL);
  }

  while ( getline(&line, &lineLen, mtab) > 0 ) 
  {
    if ( strncmp(line, cd_device, strlen(cd_device)) == 0 && line[strlen(cd_device)] == ' ' )
    {
      char *end;

      target = line + strlen(cd_device) + 1;
      for ( end = target ; *end != ' ' ; ++end )
      {
        if ( *end == '\\' ) 
        {
          char number[4];

	  memcpy(number, end + 1, 3);
          number[3] = '\0';
	  *end = (char)strtoul(number, NULL, 8);
	  memmove(end + 1, end + 4, strlen(end + 4) + 1);
        }
      }
      *end = '\0';
      break;
    }
  }

  if ( target )
    target = strdup(target);

  if ( line )
    free(line);

  return target;
}

static const cRecording *GetRecordingByName(const char *Name)
{
  const cRecording *rec = Recordings.First();
  for (; rec != NULL; rec = Recordings.Next(rec))
  {
    if (strcmp(rec->Name(), Name) == 0)
      return rec;
  }
  return NULL;
}


cCDStatus::cCDStatus(cPlugin *plugin)
  :cThread("cd drive watch thread")
{
  running = false;
  pMountPlugin = plugin;
  mediaStatus = mediaNoChange;
  m_isOpen = true;
  mounted = false;
  playing = false;
  trayLocked = false;
  cmd = cmdNop;
  fd = open(TrayopenSetup.Device, O_RDONLY | O_NONBLOCK);
}

cCDStatus::~cCDStatus()
{
  running = false;
#if VDRVERSNUM < 10332
  if (osd)
    delete osd;
#endif
  mutex.Lock();
  newCmd.Broadcast();
  mutex.Unlock();
  close(fd);
  Cancel(3);
}

bool cCDStatus::Start(void)
{											
  int status;
  if (fd > 0)
  {
    running = true;
    status = ioctl(fd, CDROM_DRIVE_STATUS, 0);
    switch(status) 
    {
      case CDS_NO_INFO: 
        break;
      case CDS_TRAY_OPEN:
#ifdef STATUSCHECK
        m_isOpen = true;
#endif
        break;
      case  CDS_DRIVE_NOT_READY: 
// is this closed but not finished disc detection ??
        break;
      case CDS_NO_DISC:
//        break;
      case CDS_DISC_OK:
        m_isOpen = false;
        break;
      default:
        //just ignore
        break;
    }

    return cThread::Start();
  }
  esyslog("Error: could CDROM invalid \"%s\".", TrayopenSetup.Device);
  return false;
}

int cCDStatus::WaitCmd(int fd, struct cdrom_generic_command *cgc, unsigned char *buffer,
  int len, int ddir, int timeout)
{
  struct request_sense sense;
  int ret;

  memset(&sense, 0, sizeof(sense));

  cgc->timeout = timeout;
  cgc->buffer = buffer;
  cgc->buflen = len;
  cgc->data_direction = ddir;
  cgc->sense = &sense;
  cgc->quiet = 1;

  ret = ioctl(fd, CDROM_SEND_PACKET, cgc);
  if (ret == -1)
  {
    esyslog("Error: could not get data from CDROM.");
  }
  return ret;
}

int cCDStatus::GetMediaEvent(int fd)
{
  struct cdrom_generic_command cgc;
  unsigned char buffer[8];
  int ret;

  memset((&cgc), 0, sizeof(struct cdrom_generic_command));
  memset(buffer, 0, sizeof(buffer));

  cgc.cmd[0] = GPCMD_GET_EVENT_STATUS_NOTIFICATION;
  cgc.cmd[1] = 1;
  cgc.cmd[4] = 16;
  cgc.cmd[8] = sizeof(buffer);

  ret = WaitCmd(fd, &cgc, buffer, sizeof(buffer), CGC_DATA_READ, 600);
  if (ret < 0)
  {
    return ret;
  }
  return buffer[4] & 0xf;
}

int cCDStatus::AllowRemoval(bool allow)
{
  struct cdrom_generic_command cgc;
  unsigned char buffer[8];
  int ret;

  memset((&cgc), 0, sizeof(struct cdrom_generic_command));
  memset(buffer, 0, sizeof(buffer));

  cgc.cmd[0] = GPCMD_PREVENT_ALLOW_MEDIUM_REMOVAL;
  cgc.cmd[4] = allow ? 0 : 1;
  cgc.cmd[8] = sizeof(buffer);

  ret = WaitCmd(fd, &cgc, buffer, sizeof(buffer), CGC_DATA_READ, 600);
  if (ret < 0)
  {
    trayLocked = !allow;
    return ret;
  }
  return buffer[4] & 0xf;
}


/*
 * Eject using SCSI commands. Return 0 if successful, 1 otherwise.
*/
int cCDStatus::EjectScsi(int fd)
{
	int status;
	struct sdata {
		int  inlen;
		int  outlen;
		char cmd[256];
	} scsi_cmd;

	scsi_cmd.inlen	= 0;
	scsi_cmd.outlen = 0;
	scsi_cmd.cmd[0] = ALLOW_MEDIUM_REMOVAL;
	scsi_cmd.cmd[1] = 0;
	scsi_cmd.cmd[2] = 0;
	scsi_cmd.cmd[3] = 0;
	scsi_cmd.cmd[4] = 0;
	scsi_cmd.cmd[5] = 0;
	status = ioctl(fd, SCSI_IOCTL_SEND_COMMAND, (void *)&scsi_cmd);
	if (status != 0)
		return 1;

	scsi_cmd.inlen	= 0;
	scsi_cmd.outlen = 0;
	scsi_cmd.cmd[0] = START_STOP;
	scsi_cmd.cmd[1] = 0;
	scsi_cmd.cmd[2] = 0;
	scsi_cmd.cmd[3] = 0;
	scsi_cmd.cmd[4] = 1;
	scsi_cmd.cmd[5] = 0;
	status = ioctl(fd, SCSI_IOCTL_SEND_COMMAND, (void *)&scsi_cmd);
	if (status != 0)
		return 1;

	scsi_cmd.inlen	= 0;
	scsi_cmd.outlen = 0;
	scsi_cmd.cmd[0] = START_STOP;
	scsi_cmd.cmd[1] = 0;
	scsi_cmd.cmd[2] = 0;
	scsi_cmd.cmd[3] = 0;
	scsi_cmd.cmd[4] = 2;
	scsi_cmd.cmd[5] = 0;
	status = ioctl(fd, SCSI_IOCTL_SEND_COMMAND, (void *)&scsi_cmd);
	if (status != 0)
		return 1;

	// force kernel to reread partition table when new disc inserted
	status = ioctl(fd, BLKRRPART);
	return (status != 0);
}



int cCDStatus::CloseTray(int fd)
{
  if ( (fd < 0) || ( ioctl(fd, CDROMCLOSETRAY, NULL) != 0 ) )
  {
    dsyslog("Error: could not eject CDROM \"%s\".", TrayopenSetup.Device);
    return(1);
  }
  m_isOpen = false;
  return(0);
}


int cCDStatus::OpenTray(int fd)
{
#if VDRVERSNUM < 10332
  if (osd)
  {
    delete osd;
    osd = NULL;
  }
#endif
//	if ( Umount() != 0)
//		return(1);

  if ( (fd < 0) || ( ioctl(fd, CDROMEJECT, NULL) != 0 ) )
  {
    if (EjectScsi(fd) != 0)
    {
      dsyslog("Error: could not eject CDROM \"%s\".", TrayopenSetup.Device);
      return(1);
    }
/*    if ( (fd < 0) || ( ioctl(fd, CDROM_LOCKDOOR, NULL) != 0 ) )
    {
        dsyslog("Error: could not unlock CDROM \"%s\".", TrayopenSetup.Device);
        return(1);
    }
    else
    {
      if ( (fd < 0) || ( ioctl(fd, CDROMEJECT, NULL) != 0 ) )
      {
        dsyslog("Error: could not eject CDROM 2.nd try\"%s\".", TrayopenSetup.Device);
        return(1);
      }
    }*/
  }
  m_isOpen = true;
  mounted = false;
  return(0);
}


int cCDStatus::Umount(void)
{
#ifdef LJ_UNMOUNT
  char *target = GetMountPoint(TrayopenSetup.Device);

  if ( target )
  {
    char *mountline;
    int   result;

    asprintf(&mountline, "umount '%s'", target);
    result = system(mountline);
    free(mountline);
    free(target);

    if ( result == -1 )
    {
      dsyslog("Error: could not umount CDROM \"%s\".", TrayopenSetup.Device);
      return(1);
    }
  }
  else if ( debug )
    dsyslog("CDROM \"%s\" is not mounted.", TrayopenSetup.Device);

  return(0);
#else
  char mountline[1032];

  snprintf(mountline, sizeof(mountline)-1, "umount %s", TrayopenSetup.Device);
  if ( system(mountline) == -1 )
  {
    if ( debug )
    {
//      perror("umount");
      dsyslog("Error: could not umount CDROM \"%s\".", TrayopenSetup.Device);
    }
    return(1);
  }
  mounted = false;
  return(0);
#endif
}


int cCDStatus::EjectRequest(void)
{
  mutex.Lock();
  cmd = cmdEject;
  newCmd.Broadcast();
  mutex.Unlock();
  return(0);
}


int cCDStatus::OpenCloseRequest(void)
{
  int status = ioctl(fd, CDROM_DRIVE_STATUS, 0);

  mutex.Lock();
  if ((status == CDS_TRAY_OPEN) || !m_isOpen)
//  if (m_isOpen)
    cmd = cmdEject;
  else
  {
    cmd = cmdLoad;
  }
  newCmd.Broadcast();
  mutex.Unlock();
  return(0);
}


int cCDStatus::StartDisc(void)
{
#if VDRVERSNUM < 10332
  if (osd)
  {
    delete osd;
    osd = NULL;
  }
#endif
  if (TrayopenSetup.LockTrayButton)
  {
    AllowRemoval(false);
  }

  if (TrayopenSetup.CDAutoStart)
  {
    if (pMountPlugin)
    {
      mounted = true;
#if VDRVERSNUM >= 10332
      cRemote::CallPlugin("vdrcd");
#else
      osd = pMountPlugin->MainMenuAction();
#endif

    }
    else
      return 1;
  }
  return 0;
}


void cCDStatus::Action()
{
  int event, first;

  mutex.Lock();
  running=true;
  first = 1;

  dsyslog("CDROM watch thread started with pid %d", getpid());

  while (running) {
    if (cmd == cmdNop)
    {
      event = GetMediaEvent(fd);
      if (event < 0)
          running = false;

      switch (event) {
        case 1:
          if ( debug ) dsyslog("eject request");
          mediaStatus = mediaReqEject;

          // hope this stops playback of the plugin
          // todo->side effect: stops a running playback
          if (!TrayopenSetup.LockTrayButton)
          {
//            if (playing)
            {
//              cControl::Control()->ProcessKey(kStop);
//              sleep(1);
              cControl::Shutdown();
              sleep(2);
            }
            if ((Umount() == 0) && trayLocked)
              AllowRemoval(true);
            OpenTray(fd);
          }
          break;
        case 2:
          if ( debug ) dsyslog("new media");
          mediaStatus = mediaNew;
          m_isOpen = false;
          StartDisc();
          break;
        case 3:
          if ( debug ) dsyslog("media removal");
          mediaStatus = mediaRemoval;
//          m_isOpen = true;
          break;
        case 4:
          if ( debug ) dsyslog("media change");
          mediaStatus = mediaChange;
          m_isOpen = false;
          StartDisc();
          break;
        case 0:
          if (first)
	  {
            if ( debug ) dsyslog("no media change");
	    break;
	  }
#ifdef STATUSCHECK
	  else
	  {
	    int status = ioctl(fd, CDROM_DRIVE_STATUS, 0);
            switch(status) 
            {
              case CDS_NO_INFO: 
                break;
              case CDS_TRAY_OPEN:
                m_isOpen = true;
                break;
              case  CDS_DRIVE_NOT_READY: 
// is this closed but not finished disc detection ??
                break;
              case CDS_NO_DISC:
                m_isOpen = false;
                if (TrayopenSetup.LockTrayButton)
                {
                  AllowRemoval(false);
                }
                break;
              case CDS_DISC_OK:
                m_isOpen = false;
                if (TrayopenSetup.LockTrayButton)
                {
                  AllowRemoval(false);
                }
                break;
              default:
                //just ignore
                break;
            }
	  }
#endif
	  break;
        default:
          if ( debug ) dsyslog("unknown media event (%d)", event);
          break;
      }
      first = 0;
    }
    else if (cmd == cmdEject)
    {
      cmd = cmdNop;

      // hope this stops playback of the plugin
      //todo->side effect: stops a running playback
//      if (playing || (Umount()!=0))
      {
//        cControl::Control()->ProcessKey(kStop);
//        sleep(1);
        cControl::Shutdown();
        sleep(2);
      }
      if ((Umount() == 0) && trayLocked)
        AllowRemoval(true);
      if (OpenTray(fd) == 0)
        err=false;
      else
        err=true;
    }
    else if (cmd == cmdLoad)
    {
      cmd = cmdNop;
      if (CloseTray(fd) == 0)
        err=false;
      else
        err=true;
    }
    newCmd.TimedWait(mutex, 500);    
  }
  dsyslog("CDROM watch thread ended");
  mutex.Unlock();
}


eMediaStatus cCDStatus::GetMediaStatus(void)
{
  return mediaStatus;
}


int cCDStatus::SetConfig(void)
{
  close(fd);
  fd = open(TrayopenSetup.Device, O_RDONLY | O_NONBLOCK);
  return(0);
}


void cCDStatus::Replaying(const cControl *control, const char *Name)
{
	if ( debug )
		printf("cText2SkinStatus::Replaying(%s)\n", Name);

	char *target = GetMountPoint(TrayopenSetup.Device);

	if (control != NULL && Name != NULL) {
		const cRecording *rec;

		//mReplayMode = replayMPlayer;
		if (strlen(Name) > 6 && Name[0]=='[' && Name[3]==']' && Name[5]=='(') {
			int i;
			for (i = 6; Name[i]; ++i) {
				if (Name[i] == ' ' && Name[i-1] == ')')
					break;
			}
			if (Name[i]) { // replaying mp3; XXX+
				//mReplayMode      = replayMP3;
				//mReplayIsLoop    = Name[1] == 'L';
				//mReplayIsShuffle = Name[2] == 'S';
			}
		} 
		else if ((rec = GetRecordingByName(Name)) && target &&
				strncmp(rec->FileName(), target, strlen(target)) == 0)
			playing = true;
		else if (strcmp(Name, "DVD") == 0)
			playing = true;
		else if (strcmp(Name, "VCD") == 0)
			playing = true;
		else if (strncmp(Name, "[image]", 7) == 0)
			;//mReplayMode = replayImage; XXX+
		else {
			char *path;

			asprintf(&path, "%s/%s", target, Name);
			if (target && access(path, F_OK) == 0)
				playing = true;
			else if (strlen(Name) > 7) {
				int i, n;
				for (i = 0, n = 0; Name[i]; ++i) {
					if (Name[i] == ' ' && Name[i-1] == ',' && ++n == 4)
						break;
				}
				if (Name[i]) // replaying DVD
					playing = true;
			}
			free(path);
		}
	} 
	else
		playing = false;

	if ( target )
		free(target);

	if ( debug )
		printf("replaying: %s\n", playing ? "true" : "false");
}
