/*
* Coverviewer plugin to VDR (C++)
*
* This code is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This code is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
* Or, point your browser to http://www.gnu.org/copyleft/gpl.html
*/

#include <time.h>

#include "control-image.h"
#include "data-image.h"
#include "setup-image.h"
#include "config.h"
#include "i18n.h"
#include "list.h"
#include "bitmap.h"
#include "parselog.h"

#include <vdr/status.h>
#include <vdr/tools.h>
#include <vdr/plugin.h>
#include <vdr/osd.h>


//////////////////////////////////////////////////////////////////////////////
/** OsdMenu
*/
cCoverMenu::cCoverMenu(const char *c_file)
:cOsdMenu(tr("Save cover"))
{
  d_file = c_file;

  Add(new cOsdItem(hk(tr("Save as cover for album"))));
  Add(new cOsdItem(hk(tr("Save as cover for track"))));
  Add(new cOsdItem(hk(tr("Save as cover for artist"))));
  
  SetHelp();
  
  Display();
}

cCoverMenu::~cCoverMenu()
{
}

void cCoverMenu::SetHelp()
{
  cOsdMenu::SetHelp(tr("Parent"),NULL,NULL,NULL);
}

void cCoverMenu::CopyAlbum(void)
{
  char *cmd;
  std::string datei;
  datei = CoverLog.log_basedir + "cover.jpg"; 
  asprintf(&cmd, "cp -f \"%s\"  \"%s\"", d_file.c_str(), datei.c_str());
  dsyslog("coverviewer: save as Album :Executing '%s'", cmd);
  system(cmd);
  free(cmd); 
}

void cCoverMenu::CopyTrack(void)
{
  char *cmd;
  std::string datei;
  datei = CoverLog.log_filename + "jpg";
  asprintf(&cmd, "cp -f \"%s\"  \"%s\"", d_file.c_str(), datei.c_str());
  dsyslog("coverviewer: save as Trackname: Executing '%s'", cmd);
  system(cmd);
  free(cmd); 
}

void cCoverMenu::CopyArtist(void)
{
  char *cmd;
  std::string datei;
  datei = CoverLog.log_coverdir + "/" + CoverLog.log_artist + ".jpg";
  asprintf(&cmd, "cp -f \"%s\"  \"%s\"", d_file.c_str(), datei.c_str());
  dsyslog("coverviewer: save as Artist: Executing '%s'", cmd);
  system(cmd);
  free(cmd); 
}




eOSState cCoverMenu::Execute(void)
{
  int ItemIndex = Current();

  if(ItemIndex == 0) {
    CopyAlbum();
  }

  if(ItemIndex == 1) {
    CopyTrack();
  }

  if(ItemIndex == 2) {
    CopyArtist();
  }
  
  return osEnd;
}


eOSState cCoverMenu::ProcessKey(eKeys Key)
{
  eOSState state = cOsdMenu::ProcessKey(Key);

  if(state == osUnknown) {
    switch(Key) {
       case kRed:
             return osBack;
	
       case kOk:
             return Execute();
	     break;
       default:
             state = osContinue;
    }
  }
  
  return state;
}

//////////////////////////////////////////////////////////////////////////////
/** C-tor Create for control-object to control the image-player
*/
cImageControl::cImageControl()
:cOsdObject()
,covermenu(NULL)
{
  m_tStarted = time(NULL);
  m_bSlideShowActiv = CoverSetup.m_bSlideShow;

  coversize = 0;
  visible=false;
  CanLoadCover=false;
  osd = NULL;

  std::string datei;
  datei = configdir;
  datei = datei + "/downloads/music_cover/";

  if(CoverLog.ParseLog()) {
  
    cFileSource *src = new cFileSource(datei.c_str());

    cDirItem *item = new cDirItem(src);
    if(item) {
      cSlideShow *newss = new cSlideShow(item);
      if(newss->Load() && newss->Count()) {
        SetSlideShow(newss);
        }
      else {
        delete newss;  
      }

    delete item;
    }
    delete src;
  }
  else {
    esyslog("coverviewer: ERROR: Could not parse logfile. Aborting!");
  }    
}


void cImageControl::SetSlideShow(cSlideShow * pNewSlideShow)
{
  theSlideShow.Assign(pNewSlideShow);
}

//////////////////////////////////////////////////////////////////////////////
/** D-tor Destroy control-object to control the image-player
*/
cImageControl::~cImageControl()
{
  if(covermenu) {delete covermenu;}
  covermenu = NULL;

  // Free OSD Data
  theSlideShow.Shutdown();
  Hide();
}

void cImageControl::Hide()
{
  if(covermenu) {
    delete covermenu;
    covermenu = NULL;
  }    

  if(osd) { delete osd; osd=NULL; }
  visible = false;
  SetNeedsFastResponse(false);
  
}


void cImageControl::Show()
{
  if (FileName()== NULL ) {
    esyslog("coverviewer : Oops can't load coverpicture : %s\n",FileName());
    }
  else {
    cCoverBitmap *coverbmp;

    bmpcolors=16;
    w=120;
    h=120;
    fh=27;
    fw=6;

    coversize=CoverSetup.Coversize;

    switch(coversize) {
          case 0:
	    bmpcolors = 248;
	    w = 120;
	    h = 120;
            depth = 8;
	    h2    = h -2*fh;
	  break;  
	  case 1:  
	    bmpcolors = 248;
	    w = 274;
	    h = 274;
            depth = 8;
	    h2    = h -2*fh;
	  break;
	  case 2:
	    bmpcolors = 248;
	    w = 300;
	    h = 256;
            depth = 8;
	    h2    = h -2*fh;
	  break;
	  case 3:
	    bmpcolors = 248;
	    w = CoverSetup.max_osdwidth;
	    h = CoverSetup.max_osdheight;
            depth = 8;
	    h2    = h -2*fh;
	  break;
    }	        

    if((coverbmp = cCoverBitmap::Load(FileName(), 255, h2, w, bmpcolors)) !=NULL) {
      Hide();
    
      CanLoadCover=true;


      if(coversize < 3) {
        osd=cOsdProvider::NewOsd( Setup.OSDLeft + ((Setup.OSDWidth /2) - (w/2)) , Setup.OSDTop + ((Setup.OSDHeight /2) - (h/2)), 19);
        }
      else {	
        osd=cOsdProvider::NewOsd(CoverSetup.max_osdleft, CoverSetup.max_osdtop, 19);
      }

      if(!osd) return;

      tArea Area[] = {{ 0, 0, w -1, h -1, depth}, };
      eOsdError result = osd->CanHandleAreas(Area, sizeof(Area) / sizeof(Area));
      if (result == oeOk) {
        osd->SetAreas(Area, sizeof(Area) / sizeof(Area));
        }
      else {
        const char *errormsg = NULL;
        switch (result) {
          case oeTooManyAreas:
            errormsg = "Too many OSD areas"; break;
          case oeTooManyColors:
            errormsg = "Too many colors"; break;
          case oeBppNotSupported:
            errormsg = "Depth not supported"; break;
          case oeAreasOverlap:
            errormsg = "Areas are overlapped"; break;
          case oeWrongAlignment:
            errormsg = "Areas not correctly aligned"; break;
          case oeOutOfMemory:
            errormsg = "OSD memory overflow"; break;
          case oeUnknown:
            errormsg = "Unknown OSD error"; break;
          default:
            break;
        } 	

        esyslog("coverviewer: ERROR! OSD open failed! Can't handle areas (%d)-%s\n", result, errormsg);
        if(osd) { delete osd; osd=0; }
        return;
      }  
 
      if(CanLoadCover) {

        osd->GetBitmap(0)->SetColor(0, tColor(0xFFFFFFFF));
        osd->GetBitmap(0)->SetColor(1, tColor(0xFF000000));
        osd->DrawRectangle(0 , 0, w -1, h -1, osd->GetBitmap(0)->Color(0));


        //Draw bar at top
        if(!CoverSetup.EnableFade) {
          char *buf;
          asprintf(&buf, "%s %i %s %i",tr("Cover:"), ImageCurrent(), tr("of"), ImageTotal());
          osd->DrawText( 0, 0, buf , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
          free(buf);
        }


        //Draw bar at bottom for menu
        osd->DrawRectangle( 0, (h -1) -fh   , w -1, h -1           , osd->GetBitmap(0)->Color(1));
        osd->DrawEllipse( 14 , (h -1) -fh +8, 26  , (h -1) - fh +21, clrRed, 0);
        osd->DrawText(    34 , (h -1) -fh   , tr("Parent"), osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), 20*fw, fh , taLeft);  
        osd->DrawEllipse( 34 + 20*fw +10 , (h -1) -fh +8, 34 + 20*fw +10 +12  , (h -1) - fh +21, clrBlue, 0);
        osd->DrawText(    34 + 20*fw + 10 +12 +6, (h -1) -fh   , tr("Menu"), osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), 20*fw, fh , taLeft);  

        for(int i = 0; i < bmpcolors; i++) {
	  osd->GetBitmap(0)->SetColor(i+4, coverbmp->Get().Color(i));
	}  

        //Draw cover
        osd->DrawBitmap( 0, fh, coverbmp->Get(), clrTransparent, clrTransparent, true);

        CanLoadCover = false;
        m_tStarted = time(NULL);

        if(CoverSetup.EnableFade) FadeIn(); else Flush();

      }

    }
    else delete coverbmp;

  }              

}


void cImageControl::FadeIn(void)
{
  SetNeedsFastResponse(true);

  int i;
  int alpha = 0;
  int fd = 15;
  
  while (!(fd==0)) {

    if(osd){
      alpha = alpha + 16;

      if(alpha > 255) alpha = 0;

      switch(fd) {
	  case 0:
             osd->DrawText( 0, 0, "" , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 15:
             osd->DrawText( 0, 0, ". Load ." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 14:
             osd->DrawText( 0, 0, ".. Load .." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 13:
             osd->DrawText( 0, 0, "... Load ..." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 12:
             osd->DrawText( 0, 0, ".... Load ...." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 11:
             osd->DrawText( 0, 0, "..... Load ....." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 10:
             osd->DrawText( 0, 0, "...... Load ......" , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 9:
             osd->DrawText( 0, 0, "....... Load ........" , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 8:
             osd->DrawText( 0, 0, "........ Load ........." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 7:
             osd->DrawText( 0, 0, "......... Load ........." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 6:
             osd->DrawText( 0, 0, ".......... Load .........." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 5:
             osd->DrawText( 0, 0, "........... Load ..........." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 4:
             osd->DrawText( 0, 0, "............ Load ............" , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 3:
             osd->DrawText( 0, 0, "............. Load ............." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 2:
             osd->DrawText( 0, 0, ".............. Load .............." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 1:
             char *buf;
             asprintf(&buf, "%s %i %s %i",tr("Cover:"), ImageCurrent(), tr("of"), ImageTotal());
             osd->DrawText( 0, 0, buf , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
             free(buf);
	     break;
	  default:
             osd->DrawText( 0, 0, "" , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
      }

      for ( i = 1; i < 256; i++) {
        osd->GetBitmap(0)->SetColor(i, (((osd->GetBitmap(0)->Color(i))<<8)>>8) | (alpha << 24));
      }

      Flush();
    }

    fd--;
  }

  Flush();

  SetNeedsFastResponse(false);
}


void cImageControl::FadeOut(void)
{
  SetNeedsFastResponse(true);

  int i;
  int alpha = 255;
  int fd = 15;
  
  while (!(fd==0)) {

    if(osd){
    
      alpha = alpha - 16;
      if(alpha < 0) alpha = 0;


      switch(fd) {
	  case 1:
             osd->DrawText( 0, 0, "" , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 2:
             osd->DrawText( 0, 0, ". Load ." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 3:
             osd->DrawText( 0, 0, ".. Load .." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 4:
             osd->DrawText( 0, 0, "... Load ..." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 5:
             osd->DrawText( 0, 0, ".... Load ...." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 6:
             osd->DrawText( 0, 0, "..... Load ....." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 7:
             osd->DrawText( 0, 0, "...... Load ......" , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 8:
             osd->DrawText( 0, 0, "....... Load ......." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 9:
             osd->DrawText( 0, 0, "........ Load ........." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 10:
             osd->DrawText( 0, 0, "......... Load .........." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 11:
             osd->DrawText( 0, 0, ".......... Load .........." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 12:
             osd->DrawText( 0, 0, "........... Load ..........." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 13:
             osd->DrawText( 0, 0, "............ Load ............" , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 14:
             osd->DrawText( 0, 0, "............. Load ............." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
	  case 15:
             osd->DrawText( 0, 0, ".............. Load .............." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
	     break;
//	  case 16:
//             osd->DrawText( 0, 0, ".............. Load .............." , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
//	     break;
	  default:
             osd->DrawText( 0, 0, "" , osd->GetBitmap(0)->Color(0), osd->GetBitmap(0)->Color(1), cFont::GetFont(fontSml), w, fh, taCenter);
      }

      for ( i = 1; i < 256; i++) {
        osd->GetBitmap(0)->SetColor(i, (((osd->GetBitmap(0)->Color(i))<<8)>>8) | (alpha << 24));
      }

      Flush();
    }

    fd--;
  }

  Flush();

  SetNeedsFastResponse(false);
}




void cImageControl::Flush(void)
{
  if(osd) osd->Flush();
}  


//////////////////////////////////////////////////////////////////////////////
/** VDR-Callback entry for Key was processed
@return eOSState
@param eKeys Key - the processed Keycode
*/
eOSState cImageControl::ProcessKey(eKeys Key)
{

  if(covermenu) {
    eOSState eOSRet = covermenu->ProcessKey(Key);
    switch(eOSRet) {
      case osBack:
              delete covermenu;
	      covermenu = NULL;
	      return osContinue;
      default: return eOSRet;
    }  	      
  }


  // Check for next image
  SlideImage();

  eOSState state = cOsdObject::ProcessKey(Key);

  if(state==osUnknown) {
    switch(Key) {

         //processed global keybindings
         // Toggle between Play/Pause
         case kPlay:     
         case kPause:   ToogleSlideShowActiv();
                       break;

         // Open menu
	 case kBlue:
			m_file = FileName();
	                Hide();
	                covermenu = new cCoverMenu(m_file.c_str());
	                break; 

         // Stop Plugin
         case kBack:
         case kRed:
         case kStop:     return osEnd;

         // Navigate between images
         case kUp:
         case kLeft:
	                if( ImageTotal() >1) {
		          PrevImage(1);
		          CanLoadCover = true;
		          Show();
		        }
		        break;
         case kDown:
         case kRight:
	                if( ImageTotal() >1) {
		          NextImage(1);
		          CanLoadCover = true;
		          Show();
		        }
		        break;
 
         case kOk:
	                break;

         default: break;
    }

  state = osContinue;
  }

  return state;

}


//////////////////////////////////////////////////////////////////////////////
/** Toogle between Play and Stop of the current SlideShow 
*/
void cImageControl::ToogleSlideShowActiv(void)
{
  m_bSlideShowActiv = !m_bSlideShowActiv;
}


//////////////////////////////////////////////////////////////////////////////
/** Check to get access for to viewed file
@return bool - false access was denied
*/
bool cImageControl::CheckAccess() const
{
  const char *szFileName = FileName();

  if(szFileName 
    && 0 == access(szFileName,F_OK))
  {
    return true;
  }        

  return false;
}


void cImageControl::OriginalImage(bool bCached)
{
  if(!CheckAccess())
  {
    esyslog("coverviewer: Operation failed");
  }
}



void cImageControl::NextImage(int Step)
{
  theSlideShow.NextImage(Step);
  OriginalImage(true);
}

void cImageControl::PrevImage(int Step)
{
  theSlideShow.PrevImage(Step);
  OriginalImage(true);
}

int cImageControl::ImageTotal(void) const
{
  return theSlideShow.ImageTotal();
}

int cImageControl::ImageCurrent(void) const
{
  return theSlideShow.ImageCurrent();
}

const char* cImageControl::FileName(void) const
{
  std::string datei;
  datei = configdir;
  datei = datei + "/coverviewer/data/nopic.png";

  if(ImageTotal() > 0) {
    cImage* pImage = theSlideShow.GetImage();
    return pImage?pImage->Name():NULL;
    }
  else {
    return datei.c_str();
  }
}

void cImageControl::SlideImage()
{
  if( m_bSlideShowActiv && ImageTotal() >0 )
  {
    if(CoverSetup.m_nSSsec <= 
        (time(NULL) - m_tStarted))
    {
      CanLoadCover = true;
      if(CoverSetup.EnableFade)  FadeOut();
      NextImage(1);
      Show();
    }
  }

}
