/*
 * webvideo.c: A plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id$
 */

#include <vdr/plugin.h>
#include <vdr/tools.h>
#include <vdr/videodir.h>
#include <vdr/i18n.h>
#include <libxml/parser.h>
#include <libxslt/xslt.h>
#include <libexslt/exslt.h>
#include <curl/curl.h>
#include <getopt.h>
#include "menu.h"
#include "service.h"
#include "download.h"
#include "mimetypes.h"
#include "common.h"

const char *VERSION               = "0.0.5";
static const char *DESCRIPTION    = trNOOP("Download video files from the web");
static const char *MAINMENUENTRY  = "Webvideo";
static const char *serviceFile    = "service.xml"; // name of a service file
char *downloaddir                 = NULL;
cMimeTypes *MimeTypes             = NULL;

class cPluginWebvideo : public cPlugin {
private:
  // Add any member variables or functions you may need here.
  cList<cVideoService> services;
  cDownloaderThread *downloadthread;

  bool LoadServices();
public:
  cPluginWebvideo(void);
  virtual ~cPluginWebvideo();
  virtual const char *Version(void) { return VERSION; }
  virtual const char *Description(void) { return tr(DESCRIPTION); }
  virtual const char *CommandLineHelp(void);
  virtual bool ProcessArgs(int argc, char *argv[]);
  virtual bool Initialize(void);
  virtual bool Start(void);
  virtual void Stop(void);
  virtual void Housekeeping(void);
  virtual void MainThreadHook(void);
  virtual cString Active(void);
  virtual const char *MainMenuEntry(void) { return MAINMENUENTRY; }
  virtual cOsdObject *MainMenuAction(void);
  virtual cMenuSetupPage *SetupMenu(void);
  virtual bool SetupParse(const char *Name, const char *Value);
  virtual bool Service(const char *Id, void *Data = NULL);
  virtual const char **SVDRPHelpPages(void);
  virtual cString SVDRPCommand(const char *Command, const char *Option, int &ReplyCode);
  };

cPluginWebvideo::cPluginWebvideo(void)
{
  // Initialize any member variables here.
  // DON'T DO ANYTHING ELSE THAT MAY HAVE SIDE EFFECTS, REQUIRE GLOBAL
  // VDR OBJECTS TO EXIST OR PRODUCE ANY OUTPUT!
  downloaddir = strdup(VideoDirectory);
}

cPluginWebvideo::~cPluginWebvideo()
{
  // Clean up after yourself!
  free(downloaddir);
  downloaddir = NULL;
}

const char *cPluginWebvideo::CommandLineHelp(void)
{
  // Return a string that describes all known command line options.
  return "  -d dir,   --downloaddir=dir  save downloaded files to dir\n";
}

bool cPluginWebvideo::ProcessArgs(int argc, char *argv[])
{
  // Implement command line argument processing here if applicable.
  static struct option long_options[] = {
    { "downloaddir",      required_argument, NULL, 'd' },
    { NULL }
  };

  int c;
  while ((c = getopt_long(argc, argv, "d:", long_options, NULL)) != -1) {
    switch (c) {
    case 'd':
      free(downloaddir);
      downloaddir = strdup(optarg);
      break;
    default:
      return false;
    }
  }
  return true;
}

bool cPluginWebvideo::Initialize(void)
{
  // Initialize any background activities the plugin shall perform.
  return true;
}

bool cPluginWebvideo::Start(void)
{
  // Start any background activities the plugin shall perform.

  // Test that run-time and compile-time libxml versions are compatible
  LIBXML_TEST_VERSION;
  xsltInit();
  exsltRegisterAll();

  cString mymimetypes = AddDirectory(ConfigDirectory(Name()), "mime.types");
  const char *mimefiles [] = {"/etc/mime.types", (const char *)mymimetypes, NULL};
  MimeTypes = new cMimeTypes(mimefiles);

  curl_global_init(CURL_GLOBAL_ALL);
  downloadthread = new cDownloaderThread();
  downloadthread->Start();

  return LoadServices();
}

void cPluginWebvideo::Stop(void)
{
  // Stop any background activities the plugin shall perform.
  delete downloadthread;
  delete MimeTypes;

  curl_global_cleanup();
  xsltCleanupGlobals();
  xmlCleanupParser();
}

void cPluginWebvideo::Housekeeping(void)
{
  // Perform any cleanup or other regular tasks.
}

void cPluginWebvideo::MainThreadHook(void)
{
  // Perform actions in the context of the main program thread.
  // WARNING: Use with great care - see PLUGINS.html!
}

cString cPluginWebvideo::Active(void)
{
  // Return a message string if shutdown should be postponed
  int c = downloadthread->GetUnfinishedCount();
  if (c > 0)
    return cString::sprintf(tr("%d downloads not finished"), c);
  else
    return NULL;
}

cOsdObject *cPluginWebvideo::MainMenuAction(void)
{
  // Perform the action when selected from the main VDR menu.
  return new cWebvideoMenu(&services, downloadthread);
}

cMenuSetupPage *cPluginWebvideo::SetupMenu(void)
{
  // Return a setup menu in case the plugin supports one.
  return NULL;
}

bool cPluginWebvideo::SetupParse(const char *Name, const char *Value)
{
  // Parse your own setup parameters and store their values.
  return false;
}

bool cPluginWebvideo::Service(const char *Id, void *Data)
{
  // Handle custom service requests from other plugins
  return false;
}

const char **cPluginWebvideo::SVDRPHelpPages(void)
{
  // Return help text for SVDRP commands this plugin implements
  return NULL;
}

cString cPluginWebvideo::SVDRPCommand(const char *Command, const char *Option, int &ReplyCode)
{
  // Process SVDRP commands this plugin implements
  return NULL;
}

bool cPluginWebvideo::LoadServices() {
  cVideoService *s;

  services.Clear();

  const char *configdir = ConfigDirectory(Name());
  if (!configdir) {
    error("Can't access config dir!");
    return false;
  }

  cReadDir d(configdir);
  if (d.Ok()) {
    struct dirent *e;
    while ((e = d.Next()) != NULL) {
      if (strcmp(e->d_name, ".") && strcmp(e->d_name, "..")) {
        cString buffer = AddDirectory(configdir, e->d_name);
        struct stat st;
        if ((stat(buffer, &st) == 0) && (S_ISDIR(st.st_mode))) {
          isyslog("Webvideo: Loading service definitions from %s", (const char *)buffer);
          s = new cVideoService();
          if (s->TryParse(buffer, serviceFile)) {
	    services.Add(s);
	  } else {
	    warning("Failed to load %s", (const char *)buffer);
	    delete s;
	  }
        }
      }
    }
    services.Sort();
  } else
    LOG_ERROR_STR(configdir);

  return true;
}

VDRPLUGINCREATOR(cPluginWebvideo); // Don't touch this!
