/*
 * parser.c: Web video plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id$
 */

#include <string.h>
#include <libxml/parser.h>
#include <libxml/xmlmemory.h>
#include <libxml/xmlstring.h>
#include <libxml/globals.h>
#include <libxml/HTMLparser.h>
#include <libxslt/transform.h>
#include <vdr/i18n.h>
#include "parser.h"
#include "menu.h"
#include "common.h"

cCharSetConv csc = cCharSetConv("UTF-8", cCharSetConv::SystemCharacterTable());

// --- cVideoInfo ----------------------------------------------------------

cVideoInfo::cVideoInfo(const char *videoUrl, const char *videoTitle) {
  url = strdup(videoUrl);
  title = strdup(videoTitle);
}

cVideoInfo::~cVideoInfo() {
  free(url);
  free(title);
}

char *cVideoInfo::GetExtensionFromURL() {
  char *ext = extensionFromUrl(url);
  if (ext)
    return ext;
  else
    return strdup("");
}

// --- cParser -------------------------------------------------------------

xmlDocPtr cParser::ApplyXSLT(const char *buffer, int size, \
                             const char *URL, const char *xsltfile) {
  // Load XSLT file
  xsltStylesheetPtr xsltdoc = xsltParseStylesheetFile(BAD_CAST xsltfile);
  if (!xsltdoc)
    return NULL;

  // Parse the downloaded HTML file
  xmlDocPtr doc = htmlReadMemory(buffer, size, \
				 URL, NULL, \
				 HTML_PARSE_NOERROR | HTML_PARSE_NOWARNING | \
				 HTML_PARSE_NONET);
  if (!doc) {
    error("htmlReadMemory failed!");

    xmlErrorPtr xmlerr =  xmlGetLastError();
    if (xmlerr) {
      error("XML error: %s", xmlerr->message);
    }

    xsltFreeStylesheet(xsltdoc);
    return NULL;
  }

  // apply the XSLT template
  xmlDocPtr parseddoc = xsltApplyStylesheet(xsltdoc, doc, NULL);
  if (!parseddoc) {
    xmlFreeDoc(doc);
    xsltFreeStylesheet(xsltdoc);
    error("Failed to apply XSLT stylesheet %s.", xsltfile);
    return NULL;
  }

  xmlFreeDoc(doc);
  xsltFreeStylesheet(xsltdoc);

  return parseddoc;
}

char *cParser::ApplyXSLTDump(const char *webpage, int size, const char *URL,
			     const char *xsltfile) {
  xmlDocPtr xmldoc = ApplyXSLT(webpage, size, URL, xsltfile);
  if (!xmldoc)
    return NULL;

  xmlChar *xmlcharbuf = NULL;
  int buflen = 0;
  xmlDocDumpMemory(xmldoc, &xmlcharbuf, &buflen);
  char *buf = (char *)malloc(buflen*sizeof(xmlChar)+1);
  memcpy(buf, xmlcharbuf, buflen*sizeof(xmlChar)+1);
  xmlFree(xmlcharbuf);

  return buf;
}

void cParser::HandleLinkNode(xmlDocPtr doc, xmlNodePtr node, xmlChar **title, xmlChar **url, eLinkType *type, xmlChar **templatefile) {
  // attributes
  *type = UNKNOWN_LINK;
  xmlChar *typeattr = xmlGetProp(node, BAD_CAST "type");
  if (!xmlStrcmp(typeattr, BAD_CAST "video"))
    *type = VIDEO_LINK;
  else if (!xmlStrcmp(typeattr, BAD_CAST "navigation-previous"))
    *type = PREV_LINK;
  else if (!xmlStrcmp(typeattr, BAD_CAST "navigation-next"))
    *type = NEXT_LINK;
  else if (!xmlStrcmp(typeattr, BAD_CAST "navigation"))
    *type = NAVIGATION_LINK;
  xmlFree(typeattr);

  xmlChar *tmp = xmlGetProp(node, BAD_CAST "template");
  *templatefile = (xmlChar *)validateFileName((char *)tmp);
  xmlFree(tmp);

  // child nodes
  *title = NULL;
  *url = NULL;
  node = node->xmlChildrenNode;
  while (node) {
    if (!xmlStrcmp(node->name, BAD_CAST "title"))
      *title = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
    else if (!xmlStrcmp(node->name, BAD_CAST "url"))
      *url = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
    node = node->next;
  }
}

eParsingError cParser::ParseVideoPage(cMemoryBuffer *webpage,
                                               const char *URL,
                                               const char *xsltfile,
                                               cVideoInfo **videoinfo)
{
  *videoinfo = NULL;

  xmlDocPtr xmldoc = ApplyXSLT(webpage->GetData(), webpage->GetLength(), URL, xsltfile);
  if (!xmldoc)
    return PARSING_XSLT_ERROR;

  // extract items and populate menu
  xmlChar *title = NULL;
  xmlChar *url = NULL;
  xmlNodePtr node = xmlDocGetRootElement(xmldoc);
  if (node)
    node = node->xmlChildrenNode;
  while (node) {
    if (!xmlStrcmp(node->name, BAD_CAST "title"))
      title = xmlNodeListGetString(xmldoc, node->xmlChildrenNode, 1);
    else if (!xmlStrcmp(node->name, BAD_CAST "url"))
      url = xmlNodeListGetString(xmldoc, node->xmlChildrenNode, 1);
    node = node->next;
  }
  if (!url) {
    xmlFreeDoc(xmldoc);
    if (title)
      xmlFree(title);
    error("Encountered a video link without URL. Ignoring.");
    return PARSING_NO_URL;
  }
  if (!title) {
    title = xmlCharStrdup(tr("<No title>"));
  }

  *videoinfo = new cVideoInfo((char *)url, csc.Convert((char *)title));

  xmlFree(url);
  xmlFree(title);
  xmlFreeDoc(xmldoc);
  return PARSING_OK;
}

eParsingError cParser::ParseASXPage(const char *asxpage, char **urlout) {
  *urlout = NULL;
  if (!asxpage)
    return PARSING_NO_URL;

  xmlDocPtr doc = htmlReadMemory(asxpage, strlen(asxpage), NULL, NULL, \
				 HTML_PARSE_NOERROR | HTML_PARSE_NOWARNING | \
				 HTML_PARSE_NONET);
  if (!doc)
    return PARSING_NO_URL;

  // Find the first <ref> tag
  xmlNodePtr node = xmlDocGetRootElement(doc);
  char *url = GetRefRecursively(node);
  if (url) {
    *urlout = url;
    return PARSING_OK;
  }

  error("Can't find REF tag in the ASX file!");
  return PARSING_NO_URL;
}

char *cParser::GetRefRecursively(xmlNodePtr node) {
  // If this is a <ref> tag and has href attribute return the
  // attribute value, otherwise recurse into the children.
  if (!xmlStrcasecmp(node->name, BAD_CAST "ref")) {
    // The parser has converted the HTML attribute names to lower
    // case, so it's safe to compare to "href"
    xmlChar *hrefattr = xmlGetProp(node, BAD_CAST "href");
    if (hrefattr) {
      int len = xmlStrlen(hrefattr);
      char *href = (char *)malloc((len+1)*sizeof(char));
      strcpy(href, (char *)hrefattr);
      xmlFree(hrefattr);
      return href;
    }
  }

  for (xmlNodePtr child = node->children; child; child = child->next) {
    char *res = GetRefRecursively(child);
    if (res)
      return res;
  }

  return NULL;
}
