/*
 * menu.c: Web video plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id$
 */

#include <stdlib.h>
#include <vdr/skins.h>
#include <vdr/tools.h>
#include <vdr/i18n.h>
#include "menu.h"
#include "service.h"
#include "download.h"
#include "parser.h"
#include "common.h"

static const char *defaultTemplate = "navigation"; // name of the default template
static const char *defaultVideoTemplate = "video"; // name of the default video page template

// --- cHistory ------------------------------------------------------------

cHistory::cHistory() {
  current = NULL;
}

void cHistory::Clear() {
  current = NULL;
  cList<cHistoryPage>::Clear();
}

void cHistory::TruncateAndAdd(cHistoryPage *page) {
  cHistoryPage *last = Last();
  while ((last) && (last != current)) {
    Del(last);
    last = Last();
  }

  Add(page);
  current = Last();
}

void cHistory::Home() {
  current = NULL;
}

cHistoryPage *cHistory::Back() {
  if (current)
    current = Prev(current);
  return current;
}

cHistoryPage *cHistory::Forward() {
  cHistoryPage *next;
  if (current) {
    next = Next(current);
    if (next)
      current = next;
  } else {
    current = First();
  }
  return current;
}


// --- cOsdLinkItem --------------------------------------------------------

cOsdLinkItem::cOsdLinkItem(const char *Title, const char *ID, const char *Url,
                           eLinkType Type, const char *TemplateFile)
: cOsdItem(Title)
{
  url = strdup(Url ? Url : "");
  type = Type;
  id = ID ? strdup(ID) : NULL;
  if (TemplateFile)
    templateFile = strdup(TemplateFile);
  else
    if (Type == VIDEO_LINK)
      templateFile = strdup(defaultVideoTemplate);
    else
      templateFile = strdup(defaultTemplate);  
}

cOsdLinkItem::~cOsdLinkItem() {
  if (id)
    free(id);
  if (url)
    free(url);
}


// --- cMenuEditStrQueryObject ---------------------------------------------

cMenuEditStrQueryObject::cMenuEditStrQueryObject(const char *Name, char *Value, int Length, const char *ID, const char *Allowed)
: cMenuEditStrItem(Name, Value, Length, Allowed)
{
  id = ID ? strdup(ID) : NULL;
  valuebuffer = Value;
  valuebufferlength = Length;
}

cMenuEditStrQueryObject::~cMenuEditStrQueryObject() {
  if (id)
    free(id);
  if (valuebuffer)
    free(valuebuffer);
}

char *cMenuEditStrQueryObject::AsQueryFragment() {
  if (id && *id && valuebuffer) {
    char *encoded = curl_easy_escape(NULL, valuebuffer, 0);
    cString tmp = cString::sprintf("%s=%s", id, encoded);
    curl_free(encoded);
    return strdup(tmp);
  }

  return NULL;
}


// --- cMenuEditStraQueryObject --------------------------------------------

cMenuEditStraQueryObject::cMenuEditStraQueryObject(const char *Name, int *Value, int NumStrings, char **Strings, char **StringIDs, const char *ID)
: cMenuEditStraItem(Name, Value, NumStrings, Strings)
{
  valueptr = value;
  numstrings = NumStrings;
  strings = Strings;
  stringids = StringIDs;
  id = ID ? strdup(ID) : NULL;
}

cMenuEditStraQueryObject::~cMenuEditStraQueryObject() {
  if (id)
    free(id);
  free(valueptr);
  for (int i=0; i<numstrings; i++) {
    free(strings[i]);
    free(stringids[i]);
  }
  free(strings);
  free(stringids);
}

char *cMenuEditStraQueryObject::AsQueryFragment() {
  if (id && *id && valueptr && (stringids[*valueptr][0] != '\0')) {
    cString tmp = cString::sprintf("%s=%s", id, stringids[*valueptr]);
    return strdup(tmp);
  }

  return NULL;
}


// --- cXMLMenu --------------------------------------------------

cXMLMenu::cXMLMenu(const char *Title, int c0, int c1, int c2,
				       int c3, int c4)
: cOsdMenu(Title, c0, c1, c2, c3, c4)
{
}

void cXMLMenu::Deserialize(xmlDocPtr doc, xmlNodePtr node) {
  if (node)
    node = node->xmlChildrenNode;
  while (node) {
    cOsdItem *item = CreateItemFromTag(doc, node);

    if (item)
      Add(item);
    else
      warning("Failed to parse menu tag: %s", (char *)node->name);

    node = node->next;
  }
}

int cXMLMenu::Load(const char *xmlstr) {
  Clear();

  xmlDocPtr xmldoc = xmlParseMemory(xmlstr, strlen(xmlstr));
  if (!xmldoc) {
    error("Failed to load menu.");

    xmlErrorPtr xmlerr =  xmlGetLastError();
    if (xmlerr) {
      error("libxml error: %s", xmlerr->message);
    }

    return -1;
  }

  Deserialize(xmldoc, xmlDocGetRootElement(xmldoc));

  xmlFreeDoc(xmldoc);
  return 0;
}


// --- cNavigationMenu -----------------------------------------------------

cNavigationMenu::cNavigationMenu(const char *Title, const char *configpath,
                                 cHistory *History, 
                                 cDownloaderThread *dlthread, 
                                 int c0, int c1, int c2, int c3, int c4)
  : cXMLMenu(Title, c0, c1, c2, c3, c4)
{
  history = History;
  downloadthread = dlthread;
  queryurl = NULL;
  querytemplate = NULL;
  SetConfigPath(configpath);
  SetHelp(tr("Back"));
}

cNavigationMenu::~cNavigationMenu() {
  if (configdir)
    free(configdir);
  if (queryurl)
    free(queryurl);
  if (querytemplate)
    free(querytemplate);
}

void cNavigationMenu::Deserialize(xmlDocPtr doc, xmlNodePtr root) {  
  // queryurl and template tags
  if (queryurl) {
    free(queryurl);
    queryurl = NULL;
  }
  if (querytemplate) {
    free(querytemplate);
    querytemplate = NULL;
  }

  xmlChar *query = xmlGetProp(root, BAD_CAST "queryurl");
  xmlChar *templatefile = xmlGetProp(root, BAD_CAST "template");
  if (query) {
    queryurl = strdup((char *)query);
    xmlFree(query);
  }
  if (templatefile) {
    querytemplate = strdup((char *)templatefile);
    xmlFree(templatefile);
  }

  cXMLMenu::Deserialize(doc, root);
}

cOsdItem *cNavigationMenu::CreateItemFromTag(xmlDocPtr doc, xmlNodePtr node) {
  cOsdItem *item = NULL;

  if (!xmlStrcmp(node->name, BAD_CAST "link")) {
    item = NewLinkItem(doc, node);
  } else if (!xmlStrcmp(node->name, BAD_CAST "textfield")) {
    item = NewTextField(doc, node);
  } else if (!xmlStrcmp(node->name, BAD_CAST "itemlist")) {
    item = NewItemList(doc, node); 
  }

  return item;
}

cOsdItem *cNavigationMenu::NewLinkItem(xmlDocPtr doc, xmlNodePtr node) {
  // type attribute
  eLinkType type = UNKNOWN_LINK;
  xmlChar *typeattr = xmlGetProp(node, BAD_CAST "type");
  if (typeattr) {
    if (!xmlStrcmp(typeattr, BAD_CAST "video"))
      type = VIDEO_LINK;
    else if (!xmlStrcmp(typeattr, BAD_CAST "navigation-previous"))
      type = PREV_LINK;
    else if (!xmlStrcmp(typeattr, BAD_CAST "navigation-next"))
      type = NEXT_LINK;
    else if (!xmlStrcmp(typeattr, BAD_CAST "navigation"))
      type = NAVIGATION_LINK;
    xmlFree(typeattr);
  }

  // template attribute
  char *templatefile = NULL;
  xmlChar *templateattr = xmlGetProp(node, BAD_CAST "template");
  if (templateattr) {
    templatefile = validateFileName((char *)templateattr);
    xmlFree(templateattr);
  }

  // title and url tags
  xmlChar *title = NULL, *url = NULL;
  node = node->xmlChildrenNode;
  while (node) {
    if (!xmlStrcmp(node->name, BAD_CAST "title")) {
      if (title)
        xmlFree(title);
      title = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
    } else if (!xmlStrcmp(node->name, BAD_CAST "url")) {
      if (url)
        xmlFree(url);
      url = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
    }
    node = node->next;
  }

  if (!title) {
    title = xmlCharStrdup(tr("<No title>"));
  }

  cOsdLinkItem *object = new cOsdLinkItem(csc.Convert((char *)title), NULL, (char *)url, type, templatefile);

  if (url)
    xmlFree(url);
  if (title)
    xmlFree(title);
  if (templatefile)
    free(templatefile);

  return object;
}

cMenuEditStrItem *cNavigationMenu::NewTextField(xmlDocPtr doc, xmlNodePtr node) {
  // id attribute
  xmlChar *id = xmlGetProp(node, BAD_CAST "id");

  // text tag
  xmlChar *text = NULL;
  node = node->xmlChildrenNode;
  while (node) {
    if (!xmlStrcmp(node->name, BAD_CAST "text")) {
      if (text)
        xmlFree(text);
      text = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
    }
    node = node->next;
  }
  if (!text)
    text = xmlCharStrdup("???");

  char *value = (char *)malloc(256*sizeof(char));
  *value = '\0';
  cMenuEditStrQueryObject *object = new cMenuEditStrQueryObject(csc.Convert((char *)text), value, 256, (char *)id);

  free(text);
  if (id)
    xmlFree(id);

  queryObjects.Append(object);
  return object;
}

cMenuEditStraItem *cNavigationMenu::NewItemList(xmlDocPtr doc, xmlNodePtr node) {
  // id attribute
  xmlChar *id = xmlGetProp(node, BAD_CAST "id");

  // text and item tags
  xmlChar *text = NULL;
  cVector<xmlChar *> items;
  cVector<xmlChar *> itemvalues;
  node = node->xmlChildrenNode;
  while (node) {
    if (!xmlStrcmp(node->name, BAD_CAST "text")) {
      if (text) {
        xmlFree(text);
        text = NULL;
      }
      text = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
    } else if (!xmlStrcmp(node->name, BAD_CAST "item")) {
      xmlChar *str = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
      if (!str)
        str = xmlCharStrdup("???");
      xmlChar *strvalue = xmlGetProp(node, BAD_CAST "value");
      if (!strvalue)
        strvalue = xmlCharStrdup("");

      items.Append(str);
      itemvalues.Append(strvalue);
    }
    node = node->next;
  }

  char **itemtable = (char **)malloc(items.Size()*sizeof(char *));
  char **itemvaluetable = (char **)malloc(items.Size()*sizeof(char *));
  for (int i=0; i<items.Size(); i++) {
    itemtable[i] = strdup(csc.Convert((char *)items.At(i)));
    itemvaluetable[i] = strdup((char *)itemvalues.At(i));
  }

  int *valueptr = (int *)malloc(sizeof(int));
  *valueptr = 0;

  cMenuEditStraQueryObject *object = new cMenuEditStraQueryObject(csc.Convert((char *)text), valueptr, items.Size(), itemtable, itemvaluetable, (char *)id);

  for (int i=0; i<items.Size(); i++) {
    xmlFree(items.At(i));
    xmlFree(itemvalues.At(i));
  }
  if(text)
    xmlFree(text);
  if (id)
    xmlFree(id);

  queryObjects.Append(object);
  return object;
}

eOSState cNavigationMenu::ProcessKey(eKeys Key)
{
  eOSState state = cXMLMenu::ProcessKey(Key);
  if (state == osBack) {
    history->Home();
  } else if (state == osUnknown) {
     switch (Key) {
     case kOk:
       if (queryurl)
         return SendQuery();
       else
         return Select();
     case kRed:
       return HistoryBack();
     case kGreen:
       return HistoryForward();
     default:
       state = osContinue;
       break;
     }
  }
  return state;
}

eOSState cNavigationMenu::SendQuery() {
  if (!queryurl) {
    error("SendQuery called without queryurl!");
    return osEnd;
  }

  char *querystr = (char *)malloc(sizeof(char)*(strlen(queryurl)+2));
  strcpy(querystr, queryurl);
  strcat(querystr, "?");

  int numparameters = 0;
  for (int i=0; i<queryObjects.Size(); i++) {
    cQueryObject *qo = queryObjects.At(i);
    char *parameter = qo->AsQueryFragment();
    if (parameter) {
      querystr = (char *)realloc(querystr, (strlen(querystr)+strlen(parameter)+2)*sizeof(char));
      if (i > 0)
	strcat(querystr, "&");
      strcat(querystr, parameter);
      numparameters++;

      free(parameter);
      parameter = NULL;
    }
  }

  if (numparameters == 0) {
    // remove the '?' because no parameters were added to the url
    querystr[strlen(querystr)-1] = '\0';
  }

  GoToURL(querystr, NAVIGATION_LINK, querytemplate);
  free(querystr);
  return osContinue;
}

eOSState cNavigationMenu::Select(void)
{
  if (!history->Current()) {
    error("Empty history in NavigationMenu!");
    return osEnd;
  }

  cOsdLinkItem *link = dynamic_cast<cOsdLinkItem *>(Get(Current()));
  if (link) {
    GoToURL(link->GetURL(), link->GetType(), link->GetTemplate());
  } else {
    debug("Selected object is not a link.");
  }
  return osContinue;
}

int cNavigationMenu::GoToURL(const char *url, eLinkType type,
                             const char *templatefile) {
  if (type == VIDEO_LINK) {
    Skins.Message(mtStatus, tr("Downloading in the background"));
  } else {
    Skins.Message(mtStatus, tr("Retrieving..."));
  }
  debug("GoToURL: %s", url);
  cMemoryBuffer *buffer;

  if (url && *url) {
    buffer = cCurlDownloader::DownloadToMemoryString(url);
  } else {
    // An empty URL should result in empty document. This is useful,
    // if the XSL transform is actually a constant XML document.
    buffer = new cMemoryBuffer(1);
    char a = '\0';
    buffer->Write(&a, 1);
  }
  if (!buffer) {
    Skins.Message(mtError, tr("Can't download web page!"));
    return -1;
  }

  cString xsltfile = cString::sprintf("%s/%s.xsl", configdir, templatefile);
  if (type == VIDEO_LINK) {
    // the URL points to a video description page
    cVideoInfo *videoinfo = NULL;
    switch (cParser::ParseVideoPage(buffer, url, xsltfile, &videoinfo)) {
    case PARSING_OK:
      DownloadVideo(videoinfo);
      delete videoinfo;
      break;
    case PARSING_NO_URL:
      Skins.Message(mtError, tr("XSLT transformation produced no URL!"));
      break;
    case PARSING_XSLT_ERROR:
      Skins.Message(mtError, tr("XSLT transformation failed."));
      break;
    default:
      Skins.Message(mtError, tr("Unknown error!"));
      break;
    }
    delete buffer;
    Skins.Message(mtStatus, NULL);
    return 0;
  } else {
    // the URL points to a navigation page
    char *buf = cParser::ApplyXSLTDump(buffer->GetData(), buffer->GetLength(), 
                                       url, xsltfile);
    delete buffer;
    if (!buf) {
      Skins.Message(mtError, tr("XSLT transformation failed."));
      return -1;
    }

    cHistoryPage *pg = new cHistoryPage(buf);
    history->TruncateAndAdd(pg);
    Populate(pg);
    Display();

    Skins.Message(mtStatus, NULL);
    return 0;
  }
}

void cNavigationMenu::SetConfigPath(const char *path) {
  configdir = strdup(path);
}

void cNavigationMenu::DownloadVideo(cVideoInfo *videoinfo) {
  downloadthread->AddRequest(videoinfo->GetURL(), downloaddir, validateFileName(videoinfo->GetTitle()));
}

void cNavigationMenu::Clear(void) {
  queryObjects.Clear(); // does not free the objects
  cOsdMenu::Clear();
}

void cNavigationMenu::Populate(cHistoryPage *page) {
  Load(page->GetXML());
}

eOSState cNavigationMenu::HistoryBack() {
  cHistoryPage *page = history->Back();
  if (page) {
    Populate(page);
    Display();
    SetHelp(tr("Back"), tr("Forward"));
    return osContinue;
  } else {
    return osBack;
  }
}

eOSState cNavigationMenu::HistoryForward() {
  Populate(history->Forward());
  Display();
  if (history->Current() == history->Last()) {
    SetHelp(tr("Back"), NULL);
  } else {
    SetHelp(tr("Back"), tr("Forward"));
  }
  return osContinue;
}


// --- cWebvideoMenu -------------------------------------------------------

cWebvideoMenu::cWebvideoMenu(cList<cVideoService> *servicelist,
			     cDownloaderThread *dlthread)
:cOsdMenu(tr("Select video source"))
{
  navigationMenu = NULL;
  history = new cHistory();
  services = servicelist;
  downloadthread = dlthread;

  for (cVideoService *service=services->First(); service; service=services->Next(service)) {
    cOsdItem *osdItem = new cOsdItem(service->GetTitle());
    Add(osdItem);
  }
  SetHelp(NULL, NULL, tr("Status"));
  Display();
}

cWebvideoMenu::~cWebvideoMenu()
{
  delete history;
}

eOSState cWebvideoMenu::Select(void)
{
  cVideoService *s = services->Get(Current());

  if (s) {
    navigationMenu = new cNavigationMenu(s->GetTitle(), s->GetConfigDir(), history, 
					 downloadthread, 25);
    navigationMenu->GoToURL(s->GetURL(), NAVIGATION_LINK, s->GetTemplate());
    return AddSubMenu(navigationMenu);
  }
  return osEnd;
}

eOSState cWebvideoMenu::ProcessKey(eKeys Key)
{
  eOSState state = cOsdMenu::ProcessKey(Key);
  if (state == osUnknown) {
     switch (Key) {
     case kOk:
       return Select();
     case kYellow:
       return AddSubMenu(new cStatusScreen(downloadthread));
     default:
       break;
     }
     state = osContinue;
  }
  return state;
}


// --- cStatusScreen -------------------------------------------------------

cStatusScreen::cStatusScreen(cDownloaderThread *thread)
  : cOsdMenu(tr("Download status"))
{
  cString countstr = cString::sprintf(tr("%d uncompleted downloads."), thread->GetUnfinishedCount());
  Add(new cOsdItem(countstr, osUnknown, false));
}
