/*
 * download.h: Web video plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id$
 */

#ifndef __WEBVIDEO_DOWNLOAD_H
#define __WEBVIDEO_DOWNLOAD_H

#include <vdr/thread.h>
#include <vdr/tools.h>
#include <curl/curl.h>

enum eDownloadError {
  DOWNLOAD_OK = 0,
  DOWNLOAD_UNKNOWN_ERROR,
  DOWNLOAD_IN_PROGRESS,
  DOWNLOAD_ABORTED,
  DOWNLOAD_CONNECTION_FAILED,
  DOWNLOAD_WRITE_ERROR,
  DOWNLOAD_UNSUPPORTED_PROTOCOL,
  DOWNLOAD_NO_MEM
};

// --- cMemoryBuffer -------------------------------------------------------

class cBuffer {
public:
  virtual ~cBuffer();
  virtual void Write(char *data, int length) = 0;
  virtual char *GetData() = 0;
  virtual int GetLength() = 0;
};

class cMemoryBuffer {
private:
  char *buf;
  size_t len;
  size_t capacity;
public:
  cMemoryBuffer(size_t prealloc = KILOBYTE(10));
  virtual ~cMemoryBuffer();

  virtual size_t Write(char *data, size_t length);
  virtual char *GetData() { return buf; }
  virtual int GetLength() { return len; }
};

// --- cCurlDownloader -----------------------------------------------------

class cCurlDownloader {
public:
  static const cString useragent;

  static CURL *CreateCurlHandle(const char *url);
  static cMemoryBuffer *DownloadToMemory(const char *url);
  static cMemoryBuffer *DownloadToMemoryString(const char *url);
};

// --- cDownloadRequest ----------------------------------------------------

class cMMSThread;

class cDownloadRequest : public cListObject {
private:
  char *url;
  char *tempfile;
  char *destbase;
  char *destdir;
  char *contenttype;
  int id;
  cUnbufferedFile *filehandle;
  CURL *curlhandle;
  cMMSThread *mmsthread;

  static int nextid;
protected:
  char *GetExtension();
public:
  cDownloadRequest(const char *requrl, const char *destdirname, const char *destbasename);
  ~cDownloadRequest();

  char *GetUrl() { return url; };
  char *GetDestDirectory() { return destdir; };
  char *GetDestBaseName() { return destbase; };
  char *GetContentType() { return contenttype; };
  int GetID() { return id; };
  CURL *GetCurlHandle() { return curlhandle; };
  cMMSThread *GetMMSThread() { return mmsthread; };
  cUnbufferedFile *GetDestFileHandle() { return filehandle; };
  char *GetFinalFileName();
  char *GetTempFileName() { return tempfile; };
  void SetUrl(const char *requrl);
  void SetCurlHandle(CURL *handle) { curlhandle = handle; };
  void SetMMSThread(cMMSThread *thread) { mmsthread = thread; };
  void SetTempFileName(const char *tmpfile);
  void SetContentType(const char *ct);
  void SetDestFileHandle(cUnbufferedFile *f);
};

// --- cDownloaderThread ---------------------------------------------------

class cDownloaderThread : public cThread {
private:
  cList<cDownloadRequest> requestList;  // currently active requests
  cList<cDownloadRequest> newRequestList; // requests not yet active
  cMutex newRequestMutex; // protects newRequestList
  cCondWait newReqCond; // signals that new requests are available
  cList<cMMSThread> mmsthreads;

  static eDownloadError CurlCodeToDlError(CURLcode curlcode);
  int HandleNewRequest(cDownloadRequest *req);
  void StartRequest(cDownloadRequest *req, CURLM *multi_handle);
  void CleanupFinishedRequests(CURLM *multi_handle);
  void CleanupRequest(cDownloadRequest *req, eDownloadError dlcode, CURLM *multi_handle);
  int ProcessASX(cDownloadRequest *req);
  eDownloadError DownloadMMS(cDownloadRequest *req);
protected:
  void Action(void);
  static size_t WriteFileCallback(void *ptr, size_t size, size_t nmemb, void *data);
  static size_t WriteHeaderCallback(void *ptr, size_t size, size_t nmemb, void *data);
public:
  cDownloaderThread();
  virtual ~cDownloaderThread();

  // Schedule a new download request to be handled by the thread.
  // Called from the main thread context.
  void AddRequest(const char *url, const char *destbasename, const char *destdirname);
  virtual void Stop(int WaitSeconds = 0);
  // Returns the number download requests currectly active
  int GetUnfinishedCount();
};

// --- cMMSThread ----------------------------------------------------------

class cMMSThread : public cThread, public cListObject {
private:
  cDownloadRequest *req;
  eDownloadError result;
  size_t (*CallbackFunc)(void *, size_t, size_t, void *);
protected:
  void Action(void);
public:
  cMMSThread(cDownloadRequest *req, size_t (*WriteCallback)(void *, size_t, size_t, void *));

  bool IsRunning();
  void Stop();
  eDownloadError Result() { return result; };
  cDownloadRequest *GetDownloadRequest() { return req; };
};

#endif // __WEBVIDEO_DOWNLOAD_H
