/*
 * download.c: Web video plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id$
 */
 
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <curl/curl.h>
#include <curl/types.h>
#include <curl/easy.h>
#ifdef USE_LOCAL_LIBMMS_HEADER
#include "libmms/mmsx.h"
#else
#include <libmms/mmsx.h>
#endif
#include <vdr/tools.h>
#include <vdr/skins.h>
#include <vdr/i18n.h>
#include "download.h"
#include "mimetypes.h"
#include "parser.h"
#include "common.h"

const cString cCurlDownloader::useragent = cString::sprintf("vdr-plugin-webvideo/%s %s", VERSION, curl_version());

static size_t WriteMemoryCallback(void *ptr, size_t size, size_t nmemb, void *data)
{
  size_t realsize = size * nmemb;
  cMemoryBuffer *buf = (cMemoryBuffer *)data;
  return buf->Write((char *)ptr, realsize);
}

#ifdef DEBUG
int curldebugcallback(CURL *curlhandle, curl_infotype type, char *msg, size_t len, void *data) {
  if ((type == CURLINFO_DATA_IN) || (type == CURLINFO_DATA_OUT))
    return 0;

  char *m = (char *)malloc(len+1);
  memcpy(m, msg, len);
  m[len] = '\0';
  debug("curl debug(%d): %s", (int)type, m);
  free(m);
  return 0;
}
#endif

// --- cMemoryBuffer -------------------------------------------------------

cMemoryBuffer::cMemoryBuffer(size_t prealloc)
{
  capacity = prealloc;
  buf = (char *)malloc(capacity);
  len = 0;
}

cMemoryBuffer::~cMemoryBuffer()
{
  if (buf)
    free(buf);
}

size_t cMemoryBuffer::Write(char *data, size_t bytes)
{
  if (len+bytes > capacity) {
    capacity += min((int)capacity, KILOBYTE(100));
    capacity = max(capacity, len+bytes);
    buf = (char *)realloc(buf, capacity);
  }

  if (buf) {
    memcpy(&buf[len], data, bytes);
    len += bytes;
    return bytes;
  }
  return 0;
}

// --- cCurlDownloader -----------------------------------------------------

CURL *cCurlDownloader::CreateCurlHandle(const char *url) {
  // Return a curl handle with common options set. The caller must
  // still set CURLOPT_WRITEFUNCTION and related options.

  // Init the curl session
  CURL *curl_handle = curl_easy_init();
  if (!curl_handle)
    return NULL;

  // No progress meter
  curl_easy_setopt(curl_handle, CURLOPT_NOPROGRESS, 1);

  // No signaling
  curl_easy_setopt(curl_handle, CURLOPT_NOSIGNAL, 1);

  // Follow HTTP redirections
  curl_easy_setopt(curl_handle, CURLOPT_FOLLOWLOCATION, 1);

  // Fail silently if the HTTP code returned is equal to or larger than 400
  curl_easy_setopt(curl_handle, CURLOPT_FAILONERROR, 1);

  // Some servers don't like requests that are made without a user-agent field
  curl_easy_setopt(curl_handle, CURLOPT_USERAGENT, (const char *)useragent);

#if 0
  curl_easy_setopt(curl_handle, CURLOPT_VERBOSE, true);
  curl_easy_setopt(curl_handle, CURLOPT_DEBUGFUNCTION, &curldebugcallback);
#endif

  // Specify URL to get
  curl_easy_setopt(curl_handle, CURLOPT_URL, url);

  return curl_handle;
}

cMemoryBuffer *cCurlDownloader::DownloadToMemory(const char *url)
{
  CURL *curl_handle = CreateCurlHandle(url);
  if (!curl_handle) {
    error("Failed to create curl handle for download %s", url);
    return NULL;
  }

  curl_easy_setopt(curl_handle, CURLOPT_TIMEOUT, 120);

  cMemoryBuffer *membuf = new cMemoryBuffer();
  curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
  curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, (void *)membuf);

  CURLcode curlres = curl_easy_perform(curl_handle);

  // Cleanup curl stuff
  curl_easy_cleanup(curl_handle);

  if (curlres != 0) {
    // Free allocated memory
    delete membuf;
    error("Failed to download %s: %d %s", url, curlres, curl_easy_strerror(curlres));
    return NULL;
  }
  
  return membuf;
}

cMemoryBuffer *cCurlDownloader::DownloadToMemoryString(const char *url) {
  cMemoryBuffer *buf = DownloadToMemory(url);
  if (!buf)
    return NULL;
  
  char a = '\0';
  buf->Write(&a, 1);
  return buf;
}

// --- cDownloadRequest ----------------------------------------------------
int cDownloadRequest::nextid = 0;

cDownloadRequest::cDownloadRequest(const char *requrl, const char *destdirname,
                                   const char *destbasename) {
  url = strdup(requrl);
  destdir = strdup(destdirname);
  destbase = strdup(destbasename);
  tempfile = NULL;
  contenttype = NULL;
  filehandle = NULL;
  id = nextid++;
}

cDownloadRequest::~cDownloadRequest() {
  free(destdir);
  free(destbase);
  free(url);
  if (contenttype)
    free(contenttype);
  if (tempfile)
    free(tempfile);
  if (filehandle)
    delete filehandle;  // close the output file
}

char *cDownloadRequest::GetExtension() {
  // Get extension from Content-Type
  char *ext = NULL;
  char *ext2 = MimeTypes->ExtensionFromMimeType(contenttype);

  // Workaround for buggy servers: If the server claims that the mime
  // type is text/plain, ignore the server and fall back to extracting
  // the extension from the URL. This function should be called only
  // for video, audio or ASX files and therefore text/plain is clearly
  // incorrect.
  if (ext2 && contenttype && !strcasecmp(contenttype, "text/plain")) {
    debug("Ignoring content type text/plain, getting extension from url.");
    free(ext2);
    ext2 = NULL;
  }

  if (ext2) {
    // Add dot in front of the extension
    ext = (char *)malloc(strlen(ext2)+2);
    ext[0] = '.';
    ext[1] = '\0';
    strcat(ext, ext2);
    free(ext2);
    return ext;
  }

  // Get extension from URL
  ext = extensionFromUrl(url);
  if (ext) {
    return ext;
  }

  // No extension!
  return strdup("");
}

char *cDownloadRequest::GetFinalFileName() {
  char *ext = GetExtension();
  cString dest = cString::sprintf("%s/%s%s", destdir, destbase, ext);

  struct stat st;
  int i = 1;
  while (stat(dest, &st) == 0) {
    dest = cString::sprintf("%s/%s-%d%s", destdir, destbase, i++, ext);
  }
  free(ext);

  if (errno == ENOENT)
    // dest is an available name
    return strdup(dest);
  else
    // Error accessing the dest dir (for example, no search
    // permissions or too many symbolic links)
    return NULL;
}

void cDownloadRequest::SetUrl(const char *requrl) {
  if (url)
    free(url);
  if (requrl)
    url = strdup(requrl);
  else
    url = NULL;
}

void cDownloadRequest::SetTempFileName(const char *tmpfile) {
  if (tempfile)
    free(tempfile);
  if (tmpfile)
    tempfile = strdup(tmpfile);
  else
    tempfile = NULL;
}

void cDownloadRequest::SetContentType(const char *ct) {
  if (contenttype)
    free(contenttype);
  if (ct)
    contenttype = strdup(ct);
  else
    contenttype = NULL;
}

void cDownloadRequest::SetDestFileHandle(cUnbufferedFile *f) {
  if (filehandle)
    delete filehandle;
  filehandle = f;
}

// --- cDownloaderThread ---------------------------------------------------

cDownloaderThread::cDownloaderThread() 
  : cThread("webvideo download")
{
}

cDownloaderThread::~cDownloaderThread() {
  if (Running())
    Stop(5);
}

int cDownloaderThread::HandleNewRequest(cDownloadRequest *req) {
  // Make sure that the destination directory exists
  char *path = req->GetDestDirectory();
  if (!DirectoryOk(path, false)) {
    isyslog("Directory %s does not exist, trying to create", path);
    if (!MakeDirs(path, true)) {
      LOG_ERROR;
      return -1;
    }
  }

  // open temporary file
  char *tmpfilename = tempnam(path, "webvideo");
  debug("temporary file: %s", tmpfilename);
  req->SetTempFileName(tmpfilename);
  cUnbufferedFile *f = new cUnbufferedFile();
  if (f->Open(tmpfilename, O_WRONLY | O_CREAT | O_EXCL, DEFFILEMODE) < 0) {
    error("Failed to create temporary file %s", tmpfilename);
    free(tmpfilename);
    delete f;
    return -1;
  }
  free(tmpfilename);
  req->SetDestFileHandle(f);

  // Check if the URL needs to be handled by libmms
  req->SetMMSThread(NULL); // Make sure we are not using old thread
                           // pointer in case this is a recycled req.
  if ((strncmp(req->GetUrl(), "mms:", 4) == 0) ||
      (strncmp(req->GetUrl(), "mmsh:", 5) == 0) || 
      (strncmp(req->GetUrl(), "mmst:", 5) == 0)) {
    cMMSThread *thread = new cMMSThread(req, WriteFileCallback);
    req->SetMMSThread(thread);
    mmsthreads.Add(thread);
    return 0;
  }

  // Create a new curl handle for a request.
  CURL *curl_handle = cCurlDownloader::CreateCurlHandle(req->GetUrl());
  if (!curl_handle) {
    req->SetDestFileHandle(NULL);
    return -1;
  }

  // Send headers to this function
  curl_easy_setopt(curl_handle, CURLOPT_HEADERFUNCTION, WriteHeaderCallback);
  curl_easy_setopt(curl_handle, CURLOPT_WRITEHEADER, req);

  // Send all data to this function
  curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, WriteFileCallback);
  curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, req);

  // Store information about the request as private data
  curl_easy_setopt(curl_handle, CURLOPT_PRIVATE, req);

  req->SetCurlHandle(curl_handle);

  return 0;
}

void cDownloaderThread::StartRequest(cDownloadRequest *req, CURLM *multi_handle) {
  cMMSThread *mmsthread = req->GetMMSThread();
  if (mmsthread) {
    mmsthread->Start();
    // From this point onward the req pointer is controlled by
    // the new thread. This thread may only access the object
    // again after the MMS thread has finished.
  } else {
    CURL *easy_handle = req->GetCurlHandle();
    if (easy_handle) {
      CURLMcode curlcode = curl_multi_add_handle(multi_handle, easy_handle);
      if (curlcode != CURLM_OK) {
        error("Failed to add handle to multi stack (CURLMcode = %d)", curlcode);
        curl_easy_cleanup(easy_handle);
        newRequestList.Del(req, true);
        return;
      }
    }
  }

  newRequestList.Del(req, false);
  requestList.Add(req);
}

void cDownloaderThread::CleanupFinishedRequests(CURLM *multi_handle) {
  CURLMsg *curlmsg;
  int msgs_in_queue;
  cDownloadRequest *req;

  do {
    curlmsg = curl_multi_info_read(multi_handle, &msgs_in_queue);
    if ((curlmsg) && (curlmsg->msg == CURLMSG_DONE)) {
      CURL *easy_handle = curlmsg->easy_handle;
      if (curl_easy_getinfo(easy_handle, CURLINFO_PRIVATE, &req) != CURLE_OK) {
        error("Failed to get priv pointer. This should not happen.");
        continue;
      }
      
      CleanupRequest(req, CurlCodeToDlError(curlmsg->data.result), multi_handle);
    }
  } while (curlmsg);

  cMMSThread *thread = mmsthreads.First(), *next;
  while (thread) {
    next = mmsthreads.Next(thread);
    if (!thread->IsRunning()) {
      // This thread has finished
      CleanupRequest(thread->GetDownloadRequest(), thread->Result(), NULL);
    }
    thread = next;
  }
}

eDownloadError cDownloaderThread::CurlCodeToDlError(CURLcode curlcode) {
  switch (curlcode) {
  case CURLE_OK: return DOWNLOAD_OK;
  case CURLE_UNSUPPORTED_PROTOCOL: return DOWNLOAD_UNSUPPORTED_PROTOCOL;
  case CURLE_OUT_OF_MEMORY: return DOWNLOAD_NO_MEM;
  default: return DOWNLOAD_CONNECTION_FAILED;
  }
}

void cDownloaderThread::Action(void) {
  int max_fd;
  fd_set read_fd_set, write_fd_set, exc_fd_set;
  CURLM *multi_handle;
  CURLMcode curlcode;
  int activerequests=0, num_handles=0, nfd;
  struct timeval timeout; 
  cDownloadRequest *req;

  multi_handle = curl_multi_init();
  if (!multi_handle) {
    error("Curl init failed. Stopping download thread.");
    return;
  }

  while (Running()) {
    if (activerequests == 0) {
      // Wait here until a download request is made
      newReqCond.Wait();
      int c = newRequestList.Count();
      if (c == 0) {
	// Signal was raised but no new request are available. This
	// happens when Cancel() was called while the thread was
	// waiting for new requests.
        if (Running())
          error("Signal was raised but no new request are available. Stopping download thread.");
	break;
      }
    }

    // if new requests are available, start them
    newRequestMutex.Lock();
    if (newRequestList.Count() > 0) {
      while (newRequestList.Count() > 0) {
	req = newRequestList.First();
        if (HandleNewRequest(req) < 0) {
          error("Failed to initialize download %s", req->GetUrl());
          newRequestList.Del(req, true);
          continue;
        }

        StartRequest(req, multi_handle);
      }

      // All new requests were handled. Clear the signal by waiting
      // until a time point in the past (that is, the cCondWait will
      // timeout immediately and clear the signal)
      newReqCond.Wait(-1);
    }
    newRequestMutex.Unlock();
    
    // The timeout for select() should be rather short so that we
    // will check if new requests are available every now and then
    // even in the case the current download has stalled.
    timeout.tv_sec = 2;
    timeout.tv_usec = 0;

    FD_ZERO(&read_fd_set);
    FD_ZERO(&write_fd_set);
    FD_ZERO(&exc_fd_set);
    curl_multi_fdset(multi_handle, &read_fd_set, &write_fd_set, &exc_fd_set, &max_fd);
    nfd = select(max_fd+1, &read_fd_set, &write_fd_set, &exc_fd_set, &timeout);
    if ((nfd < 0) && (errno != EINTR)) {
      // select error! Stop the thread.
      LOG_ERROR;
      break;
    }

    do
      curlcode = curl_multi_perform(multi_handle, &num_handles);
    while (curlcode == CURLM_CALL_MULTI_PERFORM);
    if (curlcode != CURLM_OK) {
      error("Curl error (multi code = %d)! Stopping download thread.", curlcode);
      break;
    }

    // check if some downloads have finished
    CleanupFinishedRequests(multi_handle);

    activerequests = num_handles + mmsthreads.Count();
  }

  // clean up all handles that have not finished downloading
  int nCancel = requestList.Count();
  req = requestList.First();
  while (req) {
    CleanupRequest(req, DOWNLOAD_ABORTED, multi_handle);
    req = requestList.First();
  }
  
  curl_multi_cleanup(multi_handle);

  debug("Stopping download thread");
  if (nCancel > 0)
    error("%d downloads cancelled (and %d not even started)!", nCancel, newRequestList.Count());
}

void cDownloaderThread::CleanupRequest(cDownloadRequest *req, eDownloadError dlcode, CURLM *multi_handle) {
  req->GetDestFileHandle()->Close();
  cMMSThread *mmsthread = req->GetMMSThread();
  if (mmsthread) {
    if (mmsthread->IsRunning())
      // This can happen if the download thread is forced to shutdown.
      mmsthread->Stop();
    mmsthreads.Del(mmsthread, false);
  } else {
    CURL *easy_handle = req->GetCurlHandle();
    curl_multi_remove_handle(multi_handle, easy_handle);
    curl_easy_cleanup(easy_handle);
  }

  // Special handling for ASX files. Parse and delete the temporary
  // file, and restart the download with the new url.
  char *contenttype = req->GetContentType();
  debug("Content-Type: %s", contenttype);
  if ((dlcode == DOWNLOAD_OK) && contenttype && 
      (strcmp(req->GetContentType(), "video/x-ms-asf") == 0)) {
    // Let's try to parse this as ASX file
    if (ProcessASX(req) == 0) {
      requestList.Del(req, false);
      return;
    }

    debug("Can't parse as ASX, perhaps it is ASF (audio/video file)?");
  }

  if (dlcode == DOWNLOAD_OK) {
    // move the temporary file to the final location
    char *finalname = req->GetFinalFileName();
    debug("moving %s to %s", req->GetTempFileName(), finalname);
    if ((!finalname) || (moveFile(req->GetTempFileName(), finalname) != 0))
      LOG_ERROR_STR("failed to rename the temporary file");
    if (finalname)
      free(finalname);
  } else {
    error("Failed to download %s (error = %d)", req->GetUrl(), dlcode);
    unlink(req->GetTempFileName());
  }

  requestList.Del(req);

  if (dlcode == DOWNLOAD_OK) {
    cString msg = cString::sprintf(tr("One download completed, %d remains"), requestList.Count());
    Skins.QueueMessage(mtInfo, msg, 0, -1);
    debug("%s", (const char *)msg);
  } else {
    cString msg = cString::sprintf(tr("Download failed (error = %d)"), dlcode);
    Skins.QueueMessage(mtError, msg, 0, -1);
    debug("%s", (const char *)msg);
  }
}

int cDownloaderThread::ProcessASX(cDownloadRequest *req) {
  char *buffer = readTextFile(req->GetTempFileName());
  unlink(req->GetTempFileName());
  if (!buffer)
    return -1;

  char *url = NULL;
  eParsingError err = cParser::ParseASXPage(buffer, &url);
  free(buffer);
  if (err != PARSING_OK)
    return -1;

  char *ext = extensionFromUrl(url);
  if (ext && (strcasecmp(ext, ".asx") == 0)) {
    // This playlist contains another playlist. Download it.
    req->SetUrl(url);
  } else {
    if (strncmp(url, "http:", 5) == 0) {
      debug("Changing scheme from http:// to mms://");
      char *url2 = url+1;
      memcpy(url2, "mms", 3);
      req->SetUrl(url2);
    } else {
      req->SetUrl(url);
    }
  }
  if (ext)
    free(ext);
  free(url);

  newRequestMutex.Lock();
  newRequestList.Add(req);
  newReqCond.Signal();
  newRequestMutex.Unlock();

  return 0;
}

size_t cDownloaderThread::WriteFileCallback(void *ptr, size_t size, size_t nmemb, void *data) {
  size_t realsize = size * nmemb;
  cUnbufferedFile *f = ((cDownloadRequest *)data)->GetDestFileHandle();
  return f->Write((char *)ptr, realsize);
}

size_t cDownloaderThread::WriteHeaderCallback(void *ptr, size_t size, size_t nmemb, void *data) {
  // Store the Content-Type header to the download request object
  size_t realsize = size * nmemb;
  if ((realsize >= 13) && startswith((char *)ptr, "Content-Type:")) {
    cDownloadRequest *req = (cDownloadRequest *)data;
    size_t fieldsize = realsize-13;
    char *ct = (char *)malloc(fieldsize+1);
    memcpy(ct, (char *)ptr+13, fieldsize);
    ct[fieldsize] = '\0';
    req->SetContentType(compactspace(ct));
    free(ct);
  }

  return realsize;
}

void cDownloaderThread::Stop(int WaitSeconds) {
  newReqCond.Signal(); // wake up possible sleepers  
  Cancel(WaitSeconds);
}

void cDownloaderThread::AddRequest(const char *url, const char *destdirname, 
                                   const char *destbasename) {
  // Schedule a new download request for the downloader thread. Copies
  // of the strings are stored.
  // This is called from the main thread context!
  debug("AddRequest, url: %s, destdirname: %s, destbasename: %s", url, destdirname, destbasename)
  cDownloadRequest *req = new cDownloadRequest(url, destdirname, destbasename);

  newRequestMutex.Lock();
  newRequestList.Add(req);
  newReqCond.Signal();
  newRequestMutex.Unlock();
}

int cDownloaderThread::GetUnfinishedCount() {
  // This is called from the main thread context!
  if (!Running())
    return 0;
  else
    return requestList.Count() + newRequestList.Count();
}

// --- cMMSThread ----------------------------------------------------------

cMMSThread::cMMSThread(cDownloadRequest *req, size_t (*WriteCallback)(void *, size_t, size_t, void *))
: cThread("webvideo MMS downloader")
{
  this->req = req;
  this->CallbackFunc = WriteCallback;
  result = DOWNLOAD_IN_PROGRESS;
}

bool cMMSThread::IsRunning() {
  return Running();
}

void cMMSThread::Stop() {
  Cancel(5);
}

#define MMS_BUFFER_SIZE 1024
void cMMSThread::Action(void) {
  char *buffer = (char *)malloc(MMS_BUFFER_SIZE);
  if (!buffer) {
    result = DOWNLOAD_NO_MEM;
    return;
  }

  req->SetContentType(NULL);

  mmsx_t *mms = mmsx_connect(NULL, NULL, req->GetUrl(), 1000000);
  if (!mms) {
    free(buffer);
    result = DOWNLOAD_CONNECTION_FAILED;
    return;
  }

  eDownloadError ret = DOWNLOAD_OK;
  while (Running()) {
    int n = mmsx_read(NULL, mms, buffer, MMS_BUFFER_SIZE);
    if (n == 0) {
      // done (or error?)
      break;
    }

    if (CallbackFunc(buffer, MMS_BUFFER_SIZE, 1, req) != MMS_BUFFER_SIZE) {
      ret = DOWNLOAD_WRITE_ERROR;
      break;
    }
  }
  mmsx_close(mms);
  free(buffer);

  result = ret;
  return;
}
