/*
 * common.c: Web video plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id$
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <vdr/tools.h>
#include "common.h"

char *extensionFromUrl(const char *url) {
  if (!url)
    return NULL;

  // Find the possible query ("?query=foo") or fragment ("#bar"). The
  // extension is located right before them.
  size_t extendpos = strcspn(url, "?#");

  size_t extstartpos = extendpos-1;
  while ((extstartpos > 0) && (url[extstartpos] != '.') && (url[extstartpos] != '/'))
    extstartpos--;

  if ((extstartpos > 0) && (url[extstartpos] == '.')) {
    // We found the extension. Copy it to a buffer, and return it.
    char *ext = (char *)malloc(sizeof(char)*(extendpos-extstartpos+1));
    memcpy(ext, &url[extstartpos], extendpos-extstartpos);
    ext[extendpos-extstartpos] = '\0';

    return ext;
  }

  return NULL;
}

char *validateFileName(const char *filename) {
  if (!filename)
    return NULL;

  char *validated = (char *)malloc(strlen(filename)+1);
  int j=0;
  for (unsigned int i=0; i<strlen(filename); i++) {
    if (filename[i] != '/') {
      validated[j++] = filename[i];
    }
  }
  validated[j] = '\0';
  return validated;
}

int moveFile(const char *oldpath, const char *newpath) {
  if (rename(oldpath, newpath) == 0) {
    return 0;
  } else if (errno == EXDEV) {
    // rename can't move a file between file systems. We have to copy
    // the file manually.
    int fdout = open(newpath, O_WRONLY | O_CREAT | O_EXCL, DEFFILEMODE);
    if (fdout < 0) {
      return -1;
    }

    int fdin = open(oldpath, O_RDONLY);
    if (fdin < 0) {
      close(fdout);
      return -1;
    }

    const int bufsize = 4096;
    char buffer[bufsize];
    bool ok = true;
    while (true) {
      ssize_t len = safe_read(fdin, &buffer, bufsize);
      if (len == 0) {
        break;
      } else if (len < 0) {
        ok = false;
        break;
      }

      if (safe_write(fdout, &buffer, len) != len) {
        ok = false;
        break;
      }
    }

    close(fdin);
    close(fdout);

    if (ok && (unlink(oldpath) <0)) {
      return -1;
    }

    return 0;
  } else {
    return -1;
  }
}

char *readTextFile(const char *filename) {
  long filesize;
  char *buffer;
  size_t bytesread;
  FILE *f = fopen(filename, "rb");
  if (!f)
    return NULL;

  fseek(f, 0, SEEK_END);
  filesize = ftell(f);
  rewind(f);

  buffer = (char *)malloc(sizeof(char)*filesize+1);
  if (!buffer) {
    fclose(f);
    return NULL;
  }
  bytesread = fread(buffer, 1, filesize, f);
  if ((long)bytesread != filesize) {
    fclose(f);
    free(buffer);
    return NULL;
  }
  buffer[filesize] = '\0';

  return buffer;
}
