/*
 * v4l2-i2c-drv.h - contains I2C handling code that's identical for
 *		    all V4L2 I2C drivers. Use this header if the
 *		    I2C driver is only used by drivers converted
 *		    to the bus-based I2C API.
 *
 * Copyright (C) 2007 Hans Verkuil <hverkuil@xs4all.nl>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __V4L2_I2C_DRV_H__
#define __V4L2_I2C_DRV_H__

#include <media/v4l2-common.h>

struct v4l2_i2c_driver_data {
	const char * const name;
	int driverid;
	int (*command)(struct i2c_client *client, unsigned int cmd, void *arg);
	int (*probe)(struct i2c_client *client, const struct i2c_device_id *id);
	int (*remove)(struct i2c_client *client);
	int (*suspend)(struct i2c_client *client, pm_message_t state);
	int (*resume)(struct i2c_client *client);
	int (*legacy_probe)(struct i2c_adapter *adapter);
#ifdef I2C_CLASS_TV_ANALOG
	int legacy_class;
#else
	int legacy_id;
#endif
};

static struct v4l2_i2c_driver_data v4l2_i2c_data;
static struct i2c_driver v4l2_i2c_driver;

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 26)
static int compat_legacy_probe(struct i2c_client *client)
{
	return v4l2_i2c_data.probe(client, NULL);
}
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,22)

/* Bus-based I2C implementation for kernels >= 2.6.22 */

static int __init v4l2_i2c_drv_init(void)
{
	v4l2_i2c_driver.driver.name = v4l2_i2c_data.name;
	v4l2_i2c_driver.id = v4l2_i2c_data.driverid;
	v4l2_i2c_driver.command = v4l2_i2c_data.command;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 26)
	v4l2_i2c_driver.probe = v4l2_i2c_data.probe;
#else
	v4l2_i2c_driver.probe = compat_legacy_probe;
#endif
	v4l2_i2c_driver.remove = v4l2_i2c_data.remove;
	v4l2_i2c_driver.suspend = v4l2_i2c_data.suspend;
	v4l2_i2c_driver.resume = v4l2_i2c_data.resume;
	return i2c_add_driver(&v4l2_i2c_driver);
}

#else

static struct i2c_client_address_data addr_data;

/* Bus-based I2C API is not present, add legacy code */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
static int v4l2_i2c_drv_attach_legacy(struct i2c_adapter *adapter, int address, int kind)
#else
static int v4l2_i2c_drv_attach_legacy(struct i2c_adapter *adapter, int address,
			 unsigned short flags, int kind)
#endif
{
	return v4l2_i2c_attach(adapter, address, &v4l2_i2c_driver,
			v4l2_i2c_data.name, v4l2_i2c_data.probe);
}

static int v4l2_i2c_drv_probe_legacy(struct i2c_adapter *adapter)
{
	if (v4l2_i2c_data.legacy_probe) {
		if (v4l2_i2c_data.legacy_probe(adapter))
			return i2c_probe(adapter, &addr_data, v4l2_i2c_drv_attach_legacy);
		return 0;
	}
#ifdef I2C_CLASS_TV_ANALOG
	if (adapter->class & v4l2_i2c_data.legacy_class)
#else
	if (adapter->id == v4l2_i2c_data.legacy_id)
#endif
		return i2c_probe(adapter, &addr_data, v4l2_i2c_drv_attach_legacy);
	return 0;
}

static int v4l2_i2c_drv_detach_legacy(struct i2c_client *client)
{
	int err;

	if (v4l2_i2c_data.remove)
		v4l2_i2c_data.remove(client);

	err = i2c_detach_client(client);
	if (err)
		return err;
	kfree(client);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
	MOD_DEC_USE_COUNT;
#endif
	return 0;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,14)
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,20)
static int v4l2_i2c_drv_suspend_helper(struct i2c_client *client, pm_message_t state)
#else
static int v4l2_i2c_drv_suspend_helper(struct device * dev, pm_message_t state)
#endif
#else
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,13)
static int v4l2_i2c_drv_suspend_helper(struct device * dev, pm_message_t state, u32 level)
#else
static int v4l2_i2c_drv_suspend_helper(struct device * dev, u32 state, u32 level)
#endif
#endif
{
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,20)
	struct i2c_client *client = container_of(dev, struct i2c_client, dev);
#endif
	return v4l2_i2c_data.suspend ? v4l2_i2c_data.suspend(client, state) : 0;
}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,14)
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,20)
static int v4l2_i2c_drv_resume_helper(struct i2c_client *client)
#else
static int v4l2_i2c_drv_resume_helper(struct device * dev)
#endif
#else
static int v4l2_i2c_drv_resume_helper(struct device * dev, u32 level)
#endif
{
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,20)
	struct i2c_client *client = container_of(dev, struct i2c_client, dev);
#endif
	return v4l2_i2c_data.resume ? v4l2_i2c_data.resume(client) : 0;
}
#endif

/* ----------------------------------------------------------------------- */

static struct i2c_driver v4l2_i2c_driver = {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)) && (LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,15))
	.owner = THIS_MODULE,
#endif
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,15)
	.flags = I2C_DF_NOTIFY,
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	.driver = {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,15)
		.owner = THIS_MODULE,
#endif
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,20)
		.suspend = v4l2_i2c_drv_suspend_helper,
		.resume  = v4l2_i2c_drv_resume_helper,
#endif
	},
#endif
	.attach_adapter = v4l2_i2c_drv_probe_legacy,
	.detach_client = v4l2_i2c_drv_detach_legacy,
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,20)
	.suspend = v4l2_i2c_drv_suspend_helper,
	.resume  = v4l2_i2c_drv_resume_helper,
#endif
};

/* ----------------------------------------------------------------------- */

static int __init v4l2_i2c_drv_init(void)
{
#ifdef I2C_CLASS_TV_ANALOG
	if (v4l2_i2c_data.legacy_class == 0)
		v4l2_i2c_data.legacy_class = I2C_CLASS_TV_ANALOG;
#else
	if (v4l2_i2c_data.legacy_id == 0)
		v4l2_i2c_data.legacy_id = I2C_HW_B_BT848;
#endif

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,15)
	strlcpy(v4l2_i2c_driver.name, v4l2_i2c_data.name, sizeof(v4l2_i2c_driver.name));
#else
	v4l2_i2c_driver.driver.name = v4l2_i2c_data.name;
#endif
	v4l2_i2c_driver.id = v4l2_i2c_data.driverid;
	v4l2_i2c_driver.command = v4l2_i2c_data.command;
	return i2c_add_driver(&v4l2_i2c_driver);
}

/* End legacy code */

#endif

static void __exit v4l2_i2c_drv_cleanup(void)
{
	i2c_del_driver(&v4l2_i2c_driver);
}

module_init(v4l2_i2c_drv_init);
module_exit(v4l2_i2c_drv_cleanup);

#endif /* __V4L2_I2C_DRV_H__ */
